#include "server_profile.h"

#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("bluez");

using namespace bluez_impl;

ServerProfile::ServerProfile(sdbus::IConnection& connection, std::string objectPath)
    : sdbus::AdaptorInterfaces<org::bluez::Profile1_adaptor>(connection, std::move(objectPath))
{
    registerAdaptor();
}

ServerProfile::~ServerProfile()
{
    unregisterAdaptor();
}

void ServerProfile::Release()
{
    YIO_LOG_DEBUG(__func__);
}

void ServerProfile::NewConnection(const sdbus::ObjectPath& device, const sdbus::UnixFd& fd,
                                  const std::map<std::string, sdbus::Variant>& /*fd_properties*/)
{
    YIO_LOG_INFO("New connection from " << device);
    std::thread{[fd] { Serve(fd); }}.detach();
}

void ServerProfile::RequestDisconnection(const sdbus::ObjectPath& device)
{
    YIO_LOG_INFO("Disconnection from " << device);
}

void ServerProfile::Serve(const sdbus::UnixFd& fd)
{
    try {
        SetBlocking(fd.get());
        std::string buf{};
        buf.resize(255);
        if (auto size = read(fd.get(), buf.data(), buf.size()); size != -1) {
            buf.resize(size);
        } else {
            throw std::runtime_error{fmt::format(
                "failed to read request: {}", strerror(errno))};
        }
        YIO_LOG_INFO("Eead request '" << buf << "'");
        buf = "ok";
        if (auto size = write(fd.get(), buf.data(), buf.length()); size == -1) {
            throw std::runtime_error{fmt::format(
                "failed to write response: {}", strerror(errno))};
        }
        YIO_LOG_INFO("Response sent");
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("BluezServerProfile.Serve", e.what());
    }
}

void ServerProfile::SetBlocking(int fd) noexcept {
    auto flags = fcntl(fd, F_GETFL, 0);
    flags &= ~O_NONBLOCK;
    fcntl(fd, F_SETFL, flags);
}
