#include "i_gatt_characteristics.h"

using namespace gatt_permission;

IGattCharacteristic::IGattCharacteristic(std::string name, std::string uuid, std::set<gatt_permission::GattPermission> permissions)
    : name_(std::move(name))
    , uuid_(std::move(uuid))
    , permissions_(std::move(permissions))
{
}

IGattCharacteristic::~IGattCharacteristic() = default;

const std::string& IGattCharacteristic::getName() const {
    return name_;
}

const std::string& IGattCharacteristic::getUuid() const {
    return uuid_;
}

const std::set<GattPermission>& IGattCharacteristic::getPermissions() const {
    return permissions_;
}

std::vector<uint8_t> IGattCharacteristic::readValue(const Mac& mac, uint16_t mtu, uint16_t offset) {
    if (!permissions_.count(GattPermission::READ)) {
        throw GattPermissionException(GattPermission::READ);
    }
    return readValueImpl(mac, mtu, offset);
}

void IGattCharacteristic::writeValue(const Mac& mac, const std::vector<uint8_t>& val) {
    if (!permissions_.count(GattPermission::WRITE)) {
        throw GattPermissionException(GattPermission::WRITE);
    }
    writeValueImpl(mac, val);
}

void IGattCharacteristic::enableNotifications(IGattCharacteristic::NotifyCallback cb) {
    if (cb && !permissions_.count(GattPermission::NOTIFY)) {
        throw GattPermissionException(GattPermission::NOTIFY);
    }

    cb_ = std::move(cb);
}

void IGattCharacteristic::notify() {
    if (cb_) {
        cb_();
    }
}

IGattCharacteristic::GattPermissionException::GattPermissionException(GattPermission permission)
    : std::runtime_error(std::string("No such GATT permission: ") + permissionToString(permission)) {
}
