#include "configuration.h"

#include "eval.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/json_utils/json_utils.h>

#include <iostream>
#include <memory>
#include <stdexcept>
#include <unordered_map>

using namespace quasar;

namespace YandexIO {

    Configuration::Configuration(const std::string& fileName, const ConfigPatterns& extra) {
        config_ = preprocessConfig(getConfigFromFile(fileName), extra);

        if (config_["devConfig"]["devFlag"].isString() && fileExists(config_["devConfig"]["devFlag"].asString())) {
            forceDevParams();
        }
    }

    void Configuration::forceDevParams() {
        config_["common"]["backendUrl"] = config_["devConfig"]["backendUrl"];
    }

    std::string Configuration::getDevBackendFlagPath() const {
        if (!config_["devConfig"].isMember("devFlag")) {
            throw std::runtime_error("Dev Flag is not supported by this platform");
        }
        return config_["devConfig"]["devFlag"].asString();
    }

    Configuration::Configuration(Json::Value config)
        : config_(std::move(config))
    {
    }

    Port Configuration::getServicePort(const std::string& serviceName) const {
        const auto& serviceConfig = getServiceConfig(serviceName);
        if (serviceConfig.isMember("port")) {
            return serviceConfig["port"].asInt();
        } else {
            return -1; // special value for datacratic PortRange
        }
    }

    bool Configuration::hasServiceConfig(const std::string& serviceName) const {
        return config_.isMember(serviceName) && config_[serviceName].isObject();
    }

    const Json::Value& Configuration::getServiceConfig(const std::string& serviceName) const {
        return config_[serviceName];
    }

    const Json::Value& Configuration::getCommonConfig() const {
        return getServiceConfig("common");
    }

    const Json::Value& Configuration::getFullConfig() const {
        return config_;
    }

    std::string Configuration::getDeviceType() const {
        return getString(getCommonConfig(), "deviceType");
    }

    std::string Configuration::getBackendUrl() const {
        return getString(getCommonConfig(), "backendUrl");
    }

    Json::Value& Configuration::getMutableConfig(const TestGuard& /* guard */)
    {
        return config_;
    }

    const static std::string PATTERNS = "patterns";

    Json::Value preprocessConfig(Json::Value config, const ConfigPatterns& extra) {
        if (config.isMember(PATTERNS)) {
            ConfigPatterns mp;
            Json::Value patterns = config[PATTERNS];
            for (const std::string& member : patterns.getMemberNames()) {
                if (!extra.contains(member)) {
                    Y_VERIFY(!patterns[member].isNull(), "No '%s' entry", member.c_str());
                    mp[member] = patterns[member].asString();
                } else {
                    mp[member] = extra.at(member);
                }
            }
            closure(mp);
            auto replaceFunc = [&mp](const std::string& input) {
                return bulkReplace(input, mp);
            };
            config.removeMember(PATTERNS);
            config = transform(config, replaceFunc);
        }
        return config;
    }

    std::shared_ptr<Configuration> makeConfiguration(const std::string& configPath, const ConfigPatterns& extra) {
        return std::make_shared<Configuration>(configPath, extra);
    }

    Json::Value Configuration::getSupportedFeatures() const {
        return tryGetArray(getServiceConfig("aliced"), "supportedFeatures", Json::nullValue);
    }

    bool Configuration::isFeatureSupported(const std::string& feature) const {
        const auto supportedFeatures = getSupportedFeatures();
        if (supportedFeatures.isArray()) {
            for (Json::ArrayIndex index = 0; index < supportedFeatures.size(); ++index) {
                if (supportedFeatures[index] == feature) {
                    return true;
                }
            }
        }

        return false;
    }

} // namespace YandexIO
