#include "configuration_helper.h"

using namespace quasar;

ConfigurationHelper::ConfigurationHelper(Json::Value initialConfig)
    : initialConfig_(std::move(initialConfig))
    , currentConfig_(initialConfig_)
{
}

Json::Value ConfigurationHelper::getConfigurationUpdate(const Json::Value& config) {
    std::lock_guard<std::mutex> lock(mutex_);

    Json::Value newConfig = initialConfig_;
    jsonMerge(config, newConfig);

    if (newConfig == currentConfig_) {
        return Json::Value();
    }

    auto configUpdate = getConfigDiff(currentConfig_, newConfig);
    currentConfig_ = std::move(newConfig);

    return configUpdate;
}

const Json::Value& ConfigurationHelper::getCurrentConfig() const {
    std::lock_guard<std::mutex> lock(mutex_);
    return currentConfig_;
}

Json::Value ConfigurationHelper::getConfigDiff(const Json::Value& currentConfig, const Json::Value& newConfig) {
    Json::Value diff;

    for (const auto& key : newConfig.getMemberNames()) {
        if (!currentConfig.isMember(key)) {
            diff[key] = newConfig[key];
        } else if (currentConfig[key].isObject() && newConfig[key].isObject()) {
            auto nestedDiff = getConfigDiff(currentConfig[key], newConfig[key]);
            if (!nestedDiff.isNull()) {
                diff[key] = nestedDiff;
            }
        } else if (currentConfig[key] != newConfig[key]) {
            diff[key] = newConfig[key];
        }
    }

    return diff;
}
