#include "features_config.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

using namespace quasar;

YIO_DEFINE_LOG_MODULE("configuration");

FeaturesConfig::FeaturesConfig(Json::Value fileConfig)
    : defaultConfig_(std::move(fileConfig))
{
    update();
}

FeaturesConfig::FeaturesConfig(const NAlice::TDeviceStateCapability::TState& state)
    : supportedFeatures_(state.GetSupportedFeatures().begin(), state.GetSupportedFeatures().end())
    , unSupportedFeatures_(state.GetUnsupportedFeatures().begin(), state.GetUnsupportedFeatures().end())
    , experiments_(state.GetExperiments())
{
}

std::unordered_set<TString> FeaturesConfig::featuresFromConfig(const Json::Value& config, const std::string& field) {
    std::unordered_set<TString> features;
    if (const auto& featuresJson = config[field]; featuresJson.isArray()) {
        for (const auto& feature : featuresJson) {
            if (feature.isString()) {
                features.insert(feature.asString());
            } else {
                YIO_LOG_INFO("Feature expected to be as string");
            }
        }
    }
    return features;
}

void FeaturesConfig::experimentsFromArray(const Json::Value& config, NAlice::TExperimentsProto& experiments) {
    auto* storage = experiments.MutableStorage();
    if (const auto& expsArray = config["experiments"]; expsArray.isArray()) {
        for (const auto& exp : expsArray) {
            if (exp.isString()) {
                NAlice::TExperimentsProto::TValue value;
                value.SetString("1");
                storage->insert({exp.asString(), value});
            } else {
                YIO_LOG_INFO("Experiment expected to be as string");
            }
        }
    }
}

void FeaturesConfig::experimentsFromDict(const Json::Value& config, NAlice::TExperimentsProto& experiments) {
    auto* storage = experiments.MutableStorage();
    if (const auto& expsDict = config["dictExperiments"]; expsDict.isObject()) {
        for (auto it = expsDict.begin(); it != expsDict.end(); ++it) {
            NAlice::TExperimentsProto::TValue value;
            bool success = true;
            switch (it->type()) {
                case Json::stringValue: {
                    value.SetString(it->asString());
                    break;
                }
                case Json::booleanValue: {
                    value.SetBoolean(it->asBool());
                    break;
                }
                case Json::intValue:
                case Json::uintValue: {
                    value.SetInteger(it->asInt());
                    break;
                }
                case Json::realValue: {
                    value.SetNumber(it->asDouble());
                    break;
                }
                default:
                    success = false;
                    break;
            }
            if (success) {
                storage->insert({it.name(), value});
            } else {
                YIO_LOG_WARN("Unexpected type of experiment value: " << *it);
            }
        }
    }
}

const std::unordered_set<TString>& FeaturesConfig::getSupportedFeatures() const {
    return supportedFeatures_;
}

const std::unordered_set<TString>& FeaturesConfig::getUnsupportedFeatures() const {
    return unSupportedFeatures_;
}

const NAlice::TExperimentsProto& FeaturesConfig::getExperiments() const {
    return experiments_;
}

void FeaturesConfig::addSupportedFeature(const TString& feature) {
    defaultConfig_["supportedFeatures"].append(feature);
    update();
}

void FeaturesConfig::setSupportedFeatures(const Json::Value& features) {
    defaultConfig_["supportedFeatures"] = features;
    update();
}

void FeaturesConfig::setUnsupportedFeatures(const Json::Value& features) {
    defaultConfig_["unsupportedFeautres"] = features;
    update();
}

void FeaturesConfig::setExperiments(const Json::Value& experiments) {
    if (experiments.isMember("dictExperiments")) {
        defaultConfig_["dictExperiments"] = experiments["dictExperiments"];
    }
    if (experiments.isMember("experiments")) {
        defaultConfig_["experiments"] = experiments["experiments"];
    }
    update();
}

void FeaturesConfig::processNewConfig(const Json::Value& config) {
    overrideConfig_ = config;
    update();
}

void FeaturesConfig::update() {
    if (overrideConfig_.isMember("supportedFeatures")) {
        supportedFeatures_ = featuresFromConfig(overrideConfig_, "supportedFeatures");
    } else {
        supportedFeatures_ = featuresFromConfig(defaultConfig_, "supportedFeatures");
    }

    if (overrideConfig_.isMember("unsupportedFeautres")) {
        unSupportedFeatures_ = featuresFromConfig(overrideConfig_, "unsupportedFeautres");
    } else {
        unSupportedFeatures_ = featuresFromConfig(defaultConfig_, "unsupportedFeautres");
    }

    experiments_.ClearStorage();
    if (overrideConfig_.isMember("experiments")) {
        experimentsFromArray(overrideConfig_, experiments_);
    } else {
        experimentsFromArray(defaultConfig_, experiments_);
    }
    if (overrideConfig_.isMember("dictExperiments")) {
        experimentsFromDict(overrideConfig_, experiments_);
    } else {
        experimentsFromDict(defaultConfig_, experiments_);
    }
}

void FeaturesConfig::merge(const FeaturesConfig& src) {
    supportedFeatures_.insert(src.getSupportedFeatures().begin(), src.getSupportedFeatures().end());
    unSupportedFeatures_.insert(src.getUnsupportedFeatures().begin(), src.getUnsupportedFeatures().end());
    for (const auto& [key, val] : src.getExperiments().GetStorage()) {
        experiments_.MutableStorage()->insert({key, val});
    }
}

Json::Value FeaturesConfig::formatFeatures(const std::unordered_set<TString>& features) {
    Json::Value json = Json::arrayValue;
    for (const auto& feature : features) {
        json.append(feature);
    }
    return json;
}

Json::Value FeaturesConfig::formatExperiments(const NAlice::TExperimentsProto& experiments) {
    Json::Value json = Json::objectValue;
    for (const auto& [key, val] : experiments.GetStorage()) {
        switch (val.GetValueCase()) {
            case NAlice::TExperimentsProto::TValue::kBoolean: {
                json[key] = val.GetBoolean();
                break;
            }
            case NAlice::TExperimentsProto::TValue::kInteger: {
                json[key] = val.GetInteger();
                break;
            }
            case NAlice::TExperimentsProto::TValue::kString: {
                json[key] = val.GetString();
                break;
            }
            case NAlice::TExperimentsProto::TValue::kNumber: {
                json[key] = val.GetNumber();
                break;
            }
            case NAlice::TExperimentsProto::TValue::VALUE_NOT_SET:
            default:
                break;
        }
    }
    return json;
}
