#pragma once

#include <array>
#include <list>
#include <chrono>
#include <iostream>

namespace quasar {
    /* this class tracks counters over different periods of time
     * e.g  CountersOverperiods<std::uin64_t, 60, 300, 3600, 24*3600>
     * will track counters for last minute, five minutes, hour and a day.
     * if someone call increment() every minute during a day at the end of a day
     * getCounters should return array of {1, 5, 60, 1440}
     * means 1 for last minute, 5 for last 5 minutes, 60 for last hour etc
     */

    template <typename Iter_, unsigned... Periods_>
    class ChainedCounter;

    template <typename Iter_, unsigned FirstPeriod, unsigned... Periods_>
    class ChainedCounter<Iter_, FirstPeriod, Periods_...> {
    public:
        using Counter = typename Iter_::value_type;
        using IncrementingType = typename Counter::IncrementType;

    private:
        IncrementingType total_;
        Iter_ end_;
        ChainedCounter<Iter_, Periods_...> sub_;

    public:
        ChainedCounter(Iter_ end)
            : total_{}
            , end_(end)
            , sub_(end)
        {
        }

        void fill(IncrementingType* dst, IncrementingType prev) const {
            *dst = total_;
            *dst += prev;
            sub_.fill(dst + 1, *dst);
        }

        Iter_ veryEnd(const Iter_& /*unused*/) const {
            return sub_.veryEnd(end_);
        }

        Iter_ veryEnd() const {
            return sub_.veryEnd(end_);
        }

        void newEnd(Iter_ oldEnd, Iter_ newEnd) {
            if (end_ == oldEnd) {
                end_ = newEnd;
            }
            sub_.newEnd(oldEnd, newEnd);
        }

        void increment(IncrementingType inc) {
            total_ += inc;
        }

        void maintain(std::uint64_t now, Iter_ begin, Iter_ globalEnd, IncrementingType incPart) {
            total_ += incPart;
            if (begin == end_) {
                return;
            }
            {
                IncrementingType nextIncPart{};
                auto newEnd = std::prev(end_);
                while ((now - newEnd->time) >= FirstPeriod) {
                    nextIncPart += newEnd->value;
                    total_ -= newEnd->value;
                    if (newEnd == begin) {
                        break;
                    }
                    --newEnd;
                }
                end_ = newEnd;
                incPart = nextIncPart;
            }
            // its realy safe to increment here once without checking to globalEnd, but it's not obvious so we leave that check
            if (end_ != globalEnd && now - end_->time < FirstPeriod) {
                ++end_;
            }
            sub_.maintain(now, end_, globalEnd, incPart);
        }
    };

    template <typename Iter_>
    class ChainedCounter<Iter_> {
    public:
        using Counter = typename Iter_::value_type;
        using IncrementingType = typename Counter::IncrementType;

        ChainedCounter(Iter_ /*unused*/){};
        void fill(IncrementingType* /*dst*/, IncrementingType /*prev*/) const {};
        void maintain(std::uint64_t /*now*/, Iter_ /*begin*/, Iter_ /*globalEnd*/, IncrementingType /*incPart*/){};
        Iter_ veryEnd(const Iter_& end) const {
            return end;
        }
        void newEnd(Iter_ /*oldEnd*/, Iter_ /*newEnd*/){};
    };

    /* IncrementingType have to implement:
     *   zero intialization
     *   copy constructor
     *   +=(IncrementingType) operator
     *   -=(IncrementingType) operator
     */

    template <typename IncrementingType, typename Clock, unsigned FirstPeriod, unsigned... Periods_>
    class CountersOverPeriodsImpl {
    public:
        struct Counter {
            using IncrementType = IncrementingType;
            IncrementType value;
            std::uint64_t time;
        };

    private:
        using Queue = std::list<Counter>; // FIXME: use pmr::list // deque invalidates iterators after adding :(
        Clock clock_;
        Queue queue_;
        ChainedCounter<typename Queue::iterator, FirstPeriod, Periods_...> impl_;

        void maintain(std::uint64_t now, IncrementingType head) {
            impl_.maintain(now, queue_.begin(), queue_.end(), head);
            auto oldEnd = impl_.veryEnd();
            queue_.erase(oldEnd, queue_.end());
            impl_.newEnd(oldEnd, queue_.end());
        }

    public:
        CountersOverPeriodsImpl()
            : impl_(queue_.end())
        {
        }

        CountersOverPeriodsImpl(const Clock& clock)
            : clock_(clock)
            , impl_(queue_.end())
        {
        }

        static const unsigned Size = 1 + sizeof...(Periods_);
        using Result = std::array<IncrementingType, Size>;

        // default parameter work only for integral IncrementingType
        void increment(IncrementingType inc = 1) {
            const auto now = clock_.getNow();
            if (!queue_.empty() && queue_.front().time == now) {
                queue_.front().value += inc;
                impl_.increment(inc);
            } else {
                queue_.push_front({.value = inc, .time = now});
                maintain(now, inc);
            }
        }

        Result getCounters() const {
            Result result;
            impl_.fill(result.data(), {});
            return result;
        }

        Result getUpdatedCounters() {
            maintain(clock_.getNow(), {});
            Result result;
            impl_.fill(result.data(), {});
            return result;
        }
    };

    struct SteadyClock {
        static std::uint64_t getNow();
    };

    struct SteadyClock5min {
        static std::uint64_t getNow();
    };

    template <typename IncrementingType, unsigned... Periods_>
    using CountersOverPeriods = CountersOverPeriodsImpl<IncrementingType, SteadyClock, Periods_...>;
} // namespace quasar
