#include "symmetric.h"

#include <openssl/conf.h>
#include <openssl/err.h>
#include <openssl/evp.h>

#include <cstdint>
#include <mutex>

namespace {
    class AESCrypto256: public quasar::SymmetricCrypto {
        struct EVP_CIPHER_CTX_Deleter {
            void operator()(EVP_CIPHER_CTX* ptr) {
                EVP_CIPHER_CTX_free(ptr);
            }
        };
        using EVP_CIPHER_CTX_Ptr = std::unique_ptr<EVP_CIPHER_CTX, EVP_CIPHER_CTX_Deleter>;
        static constexpr unsigned KeySize = 256 / 8;
        static constexpr unsigned BlockSize = 128 / 8;
        std::uint8_t key_[KeySize], iv_[BlockSize];
        EVP_CIPHER_CTX_Ptr ctx;

        static EVP_CIPHER_CTX_Ptr makePtr() {
            return EVP_CIPHER_CTX_Ptr(EVP_CIPHER_CTX_new());
        }

        void setupKey(const std::string& secret) {
            auto filled = std::min(KeySize, unsigned(secret.length()));
            memcpy(key_, secret.data(), filled);
            for (unsigned i = filled; i < KeySize; ++i) {
                key_[i] = 0;
            };
            if (secret.length() > KeySize) { // fill iv_
            };
        }

    public:
        AESCrypto256(const std::string& secret)
            : iv_{0x66, 0xf4, 0x61, 0x78, 0x72, 0xda, 0x6e, 0xe5, 0xcf, 0x67, 0x8a, 0x82, 0x28, 0x20, 0x25, 0x08}
            , // just junk
            ctx{makePtr()}
        {
            setupKey(secret);
        }

        std::string encrypt(const std::string& src) override {
            if (EVP_EncryptInit_ex(ctx.get(), EVP_aes_256_cbc(), nullptr, key_, iv_) != 1) {
                throw std::runtime_error("Failed to EVP_EncryptInit");
            }
            std::string rval(src.size() + BlockSize, ' ');
            int rvalLen = rval.size();
            if (EVP_EncryptUpdate(ctx.get(), (std::uint8_t*)rval.data(), &rvalLen, (const std::uint8_t*)src.data(), (int)src.size()) != 1) {
                throw std::runtime_error("Failed to EVP_EncryptUpdate");
            }
            const int mostLen = rvalLen;
            if (EVP_EncryptFinal_ex(ctx.get(), (std::uint8_t*)rval.data() + rvalLen, &rvalLen) != 1) {
                throw std::runtime_error("Failed to EVP_EncryptFinal");
            }
            rval.resize(mostLen + rvalLen);
            return rval;
        }

        std::string decrypt(const std::string& src) override {
            if (src.size() % BlockSize) {
                throw std::runtime_error("Not padded encrypted data");
            }
            if (EVP_DecryptInit_ex(ctx.get(), EVP_aes_256_cbc(), nullptr, key_, iv_) != 1) {
                throw std::runtime_error("Failed to EVP_DecryptInit");
            }
            std::string rval(src.size(), ' ');
            int rvalLen = rval.size();
            if (EVP_DecryptUpdate(ctx.get(), (std::uint8_t*)rval.data(), &rvalLen, (const std::uint8_t*)src.data(), (int)src.size()) != 1) {
                throw std::runtime_error("Failed to EVP_DecryptUpdate");
            }
            if (rvalLen == (int)src.size()) {
                return rval;
            }
            const int mostLen = rvalLen;
            if (EVP_DecryptFinal_ex(ctx.get(), (std::uint8_t*)rval.data() + rvalLen, &rvalLen) != 1) {
                throw std::runtime_error("Failed to EVP_DecryptFinal");
            }
            rval.resize(mostLen + rvalLen);
            return rval;
        }

        void changeSecret(const std::string& secret) override {
            setupKey(secret);
        }
    };

    // extract it to header when needed
    template <typename Crypto_>
    class LockedCrypto: public Crypto_ {
        std::mutex mutex_;
        using Guard = std::lock_guard<std::mutex>;

    public:
        template <typename... Params_>
        LockedCrypto(Params_... params)
            : Crypto_(params...)
                  {};

        std::string encrypt(const std::string& src) override {
            Guard lock(mutex_);
            return Crypto_::encrypt(src);
        }

        std::string decrypt(const std::string& src) override {
            Guard lock(mutex_);
            return Crypto_::decrypt(src);
        }

        void changeSecret(const std::string& secret) override {
            Guard lock(mutex_);
            Crypto_::changeSecret(secret);
        }
    };

    class NullCrypto: public quasar::SymmetricCrypto {
    public:
        std::string encrypt(const std::string& src) override {
            return src;
        }

        std::string decrypt(const std::string& src) override {
            return src;
        }

        void changeSecret(const std::string& /*secret*/) override {
        }
    };
} // namespace

namespace quasar {
    SymmetricCrypto::UniquePtr SymmetricCrypto::makeNull() {
        return std::make_unique<NullCrypto>();
    }

    SymmetricCrypto::UniquePtr SymmetricCrypto::makeAES256(const std::string& cryptoKey) {
        return std::make_unique<AESCrypto256>(cryptoKey);
    }

    SymmetricCrypto::UniquePtr SymmetricCrypto::makeAES256Locked(const std::string& cryptoKey) {
        return std::make_unique<LockedCrypto<AESCrypto256>>(cryptoKey);
    }

} // namespace quasar
