#pragma once

#include <memory>
#include <string>
#include <string_view>

struct rsa_st;
using RSA = struct rsa_st;

namespace quasar {
    class Cryptography {
    public:
        enum class Padding {
            RSA_PKCS1,
            RSA_PKCS1_OAEP
        };

        static Padding parsePadding(int value);
        static int paddingValue(Padding padding);

        struct KeyPair {
            std::string publicKey;
            std::string privateKey;

            static KeyPair fromFiles(const std::string& publicKeyPath, const std::string& privateKeyPath);
            static KeyPair fromPrivateKeyFile(const std::string& privateKeyPath);
        };

        class RSADeleter {
        public:
            void operator()(RSA* rsa);
        };

        void loadPublicKey(const std::string& fileName);  // deprecated
        void loadPrivateKey(const std::string& fileName); // deprecated

        // XXX: Not exception-safe. In case of a key loading error, object state
        // is left inconsistent.
        void setKeyPair(const KeyPair& keys);

        void setPublicKey(std::string_view encryptionKey);
        void setPrivateKey(std::string_view privateKey);

        std::string encrypt(std::string_view data, Padding padding = Padding::RSA_PKCS1_OAEP) const;
        std::string decrypt(std::string_view data, Padding padding = Padding::RSA_PKCS1_OAEP) const;

        static std::string decryptAES(std::string_view iv, std::string_view key, std::string_view data);

        std::string sign(std::string_view data) const;
        bool checkSignature(std::string_view data, std::string_view sign) const;

        static std::string hashWithHMAC_SHA256(std::string_view data, std::string_view key);

        std::string generateAESKeyString() const;

    private:
        std::unique_ptr<RSA, RSADeleter> publicRSA_;
        std::unique_ptr<RSA, RSADeleter> privateRSA_;
    };

} // namespace quasar
