#include "digest.h"

#include <yandex_io/libs/base/base58.h>
#include <yandex_io/libs/base/utils.h>

#include <contrib/libs/xxhash/xxhash.h>

#include <openssl/md5.h>
#include <openssl/sha.h>

#include <iomanip>
#include <sstream>

namespace quasar {

    std::string calcMD5Digest(std::string_view s) {
        unsigned char data[MD5_DIGEST_LENGTH];
        ::MD5((const unsigned char*)s.data(), s.size(), data);

        std::string result;
        result.reserve(32);
        for (int i = 0; i < MD5_DIGEST_LENGTH; i++) {
            char c[8];
            snprintf(c, 8, "%02x", data[i]);
            result += c;
        }
        return result;
    }

    std::string calcSHA1Digest(std::string_view data) {
        unsigned char result[SHA_DIGEST_LENGTH];
        ::SHA1((const unsigned char*)data.data(), data.length(), result);
        std::stringstream resultStr;

        resultStr << std::setfill('0');
        for (int i = 0; i < SHA_DIGEST_LENGTH; ++i) {
            resultStr << std::hex << std::setw(2) << uint32_t(result[i]);
        }

        return resultStr.str();
    }

    Md5Hasher::Md5Hasher()
    {
        static_assert(std::tuple_size<decltype(md5ctx_)>::value == sizeof(MD5_CTX));
        MD5_Init(reinterpret_cast<MD5_CTX*>(md5ctx_.data()));
    }

    Md5Hasher& Md5Hasher::update(const void* data, size_t size)
    {
        MD5_CTX* ctx = reinterpret_cast<MD5_CTX*>(md5ctx_.data());
        if (!invalidated_) {
            invalidated_ = true;
            hash_ = decltype(hash_){};
            MD5_Init(ctx);
        }
        MD5_Update(ctx, data, size);
        return *this;
    }

    Md5Hasher& Md5Hasher::update(const std::string& text)
    {
        return update(text.data(), text.size());
    }

    Md5Hasher& Md5Hasher::update(std::string_view text)
    {
        return update(text.data(), text.size());
    }

    Md5Hasher& Md5Hasher::update(const char* text)
    {
        return update(text, strlen(text));
    }

    Md5Hasher& Md5Hasher::finalize()
    {
        static_assert(std::tuple_size<decltype(hash_)>::value == MD5_DIGEST_LENGTH);
        MD5_Final(hash_.data(), reinterpret_cast<MD5_CTX*>(md5ctx_.data()));
        invalidated_ = false;
        return *this;
    }

    Md5Hash Md5Hasher::hash() const {
        return hash_;
    }

    std::string Md5Hasher::hashString(Encoder encoder, size_t bytes) const {
        bytes = (bytes ? std::min(hash_.size(), bytes) : hash_.size());
        switch (encoder)
        {
            case Encoder::HEX: {
                const char* symbols = "0123456789abcdef";
                std::string text;
                text.reserve(bytes * 2);
                for (size_t i = 0; i < bytes; ++i) {
                    text += symbols[hash_[i] >> 4];
                    text += symbols[hash_[i] & 0x0F];
                }
                return text;
            }
            case Encoder::BASE64:
                return base64Encode(reinterpret_cast<const char*>(hash_.data()), bytes);
            case Encoder::BASE58:
                return base58Encode(hash_.data(), bytes);
        }
        return "";
    }

    XXH64Hasher::XXH64Hasher(uint64_t seed) {
        static_assert(std::tuple_size<decltype(state_)>::value == 88);
        XXH64_reset(reinterpret_cast<XXH64_state_t*>(state_.data()), seed);
    }

    XXH64Hasher& XXH64Hasher::update(const void* data, size_t size)
    {
        XXH64_update(reinterpret_cast<XXH64_state_t*>(state_.data()), data, size);
        return *this;
    }

    XXH64Hasher& XXH64Hasher::update(const std::string& text)
    {
        return update(text.data(), text.size());
    }

    XXH64Hasher& XXH64Hasher::update(std::string_view text)
    {
        return update(text.data(), text.size());
    }

    XXH64Hasher& XXH64Hasher::update(const char* text)
    {
        return update(text, strlen(text));
    }

    uint64_t XXH64Hasher::hash() const {
        return XXH64_digest(reinterpret_cast<const XXH64_state_t*>(state_.data()));
    }

} // namespace quasar
