#pragma once

#include <array>
#include <cstdint>
#include <string>
#include <string_view>
#include <type_traits>

namespace quasar {
    using Md5Hash = std::array<uint8_t, 16>;

    std::string calcMD5Digest(std::string_view s);
    std::string calcSHA1Digest(std::string_view data);

    class Md5Hasher {
    public:
        Md5Hasher();
        Md5Hasher& update(const void* data, size_t size);
        Md5Hasher& update(const std::string& text);
        Md5Hasher& update(std::string_view text);
        Md5Hasher& update(const char* text);
        template <class T>
        Md5Hasher& update(const T& podValue) {
            static_assert(std::is_pod<T>::value);
            static_assert(!std::is_pointer<decltype(podValue)>::value);
            return update(&podValue, sizeof(podValue));
        }
        Md5Hasher& finalize();
        Md5Hash hash() const;

        enum class Encoder {
            HEX,
            BASE64,
            BASE58,
        };
        std::string hashString(Encoder /*encoder*/ = Encoder::HEX, size_t bytes = 0) const;

    private:
        static constexpr size_t ctxsize = sizeof(unsigned int) * (4 + 2 + 16 + 1);
        std::array<uint8_t, ctxsize> md5ctx_{};
        Md5Hash hash_{};
        bool invalidated_{true};
    };

    class XXH64Hasher {
    public:
        XXH64Hasher(uint64_t seed = 0);
        XXH64Hasher& update(const void* data, size_t size);
        XXH64Hasher& update(const std::string& text);
        XXH64Hasher& update(std::string_view text);
        XXH64Hasher& update(const char* text);
        template <class T>
        XXH64Hasher& update(const T& podValue) {
            static_assert(std::is_pod<T>::value);
            static_assert(!std::is_pointer<decltype(podValue)>::value);
            return update(&podValue, sizeof(podValue));
        }
        uint64_t hash() const;

    private:
        static constexpr size_t ctxsize = (sizeof(uint64_t) * 11);
        std::array<uint8_t, ctxsize> state_{};
    };
} // namespace quasar
