#include "delay_timings_policy.h"

#include <algorithm>

namespace quasar {

    BackoffRetriesWithRandomPolicy::BackoffRetriesWithRandomPolicy()
        : defaultCheckPeriod_(0)
        , spedUpCheckPeriod_(0)
        , maxCheckPeriod_(0)
        , minCheckPeriod_(0)
        , maxDelayMultiplier_(DEFAULT_DELAY_MULTIPLIER)
    {
    }

    BackoffRetriesWithRandomPolicy::BackoffRetriesWithRandomPolicy(const uint32_t seed,
                                                                   const int maxDelayMultiplier)
        : defaultCheckPeriod_(0)
        , spedUpCheckPeriod_(0)
        , maxCheckPeriod_(0)
        , minCheckPeriod_(0)
        , maxDelayMultiplier_(maxDelayMultiplier)
    {
        // Seed depends on device_id to guarantee different devices to update in different moment even if boot at the same millisecond
        randomGenerator_.seed(seed);
    }

    void BackoffRetriesWithRandomPolicy::initCheckPeriod(const std::chrono::milliseconds defaultCheckPeriod,
                                                         const std::chrono::milliseconds spedUpCheckPeriod,
                                                         const std::optional<std::chrono::milliseconds> maxCheckPeriod,
                                                         const std::optional<std::chrono::milliseconds> minCheckPeriod) {
        defaultCheckPeriod_ = defaultCheckPeriod;
        spedUpCheckPeriod_ = spedUpCheckPeriod;
        currentCheckPeriod_ = defaultCheckPeriod_;
        maxCheckPeriod_ = maxCheckPeriod.value_or(defaultCheckPeriod_);
        minCheckPeriod_ = minCheckPeriod.value_or(std::chrono::milliseconds(0));
    }

    void BackoffRetriesWithRandomPolicy::updateCheckPeriod(const std::optional<std::chrono::milliseconds> defaultCheckPeriod,
                                                           const std::optional<std::chrono::milliseconds> spedUpCheckPeriod) {
        currentCheckPeriod_ = std::clamp(defaultCheckPeriod.value_or(currentCheckPeriod_), minCheckPeriod_, maxCheckPeriod_);
        defaultCheckPeriod_ = std::clamp(defaultCheckPeriod.value_or(defaultCheckPeriod_), minCheckPeriod_, maxCheckPeriod_);
        spedUpCheckPeriod_ = std::clamp(spedUpCheckPeriod.value_or(spedUpCheckPeriod_), minCheckPeriod_, maxCheckPeriod_);
    }

    /**
     * First retry should be after 10 sec after timeout, second in 1 min, third in 5 min.
     * To make it more flexible retry delay consists of two parts: const and random.
     * If const part is X, random is uniformly distributed on (0, 2X).
     * @return time in ms depending on retry number.
     */
    std::chrono::milliseconds BackoffRetriesWithRandomPolicy::calcHttpClientRetriesDelay(const int retryNum) {
        std::chrono::milliseconds basicDelay{0};
        switch (retryNum) {
            case 0:
                basicDelay = std::chrono::seconds(5); // to have 10s on average
                break;
            case 1:
                basicDelay = std::chrono::seconds(30); // to have 1min on average
                break;
            case 2:
                basicDelay = std::chrono::seconds(150); // to have 5min on average
                break;
            default:
                basicDelay = std::chrono::seconds(300); // seems strange to have so many retries. do not recalc further
                break;
        }
        std::uniform_int_distribution<ssize_t> distribution(0, basicDelay.count() * 2);
        return basicDelay + std::chrono::milliseconds{distribution(randomGenerator_)};
    }

    std::chrono::milliseconds BackoffRetriesWithRandomPolicy::getDelayBetweenCalls() {
        std::uniform_int_distribution<ssize_t> distribution(0, currentCheckPeriod_.count() * maxDelayMultiplier_);
        return currentCheckPeriod_ + std::chrono::milliseconds(distribution(randomGenerator_));
    }

    std::mt19937& BackoffRetriesWithRandomPolicy::getRandomGenerator() {
        return randomGenerator_;
    };

    void BackoffRetriesWithRandomPolicy::resetDelayBetweenCallsToDefault() {
        currentCheckPeriod_ = defaultCheckPeriod_;
    }

    void BackoffRetriesWithRandomPolicy::spedUpDelayBetweenCalls() {
        currentCheckPeriod_ = spedUpCheckPeriod_;
    }

    void BackoffRetriesWithRandomPolicy::increaseDelayBetweenCalls() {
        currentCheckPeriod_ = std::min(currentCheckPeriod_ * 2, maxCheckPeriod_);
    }

} /* namespace quasar */
