#include "device.h"

#include <yandex_io/libs/json_utils/json_utils.h>

#include <mutex>
#include <shared_mutex>
#include <string_view>

using namespace YandexIO;

namespace {
    std::shared_mutex deviceMutex_;
    std::shared_ptr<IDevice> device_;
    constexpr const char* DEFAULT_USER_AGENT = "Mozilla\\/5.0 (Linux; Android 6.0.1; Station Build\\/MOB30J; wv) AppleWebKit\\/537.36 (KHTML, like Gecko) Version\\/4.0 Chrome\\/61.0.3163.98 Safari\\/537.36 YandexStation\\/2.3.0.3.373060213.20190204.develop.ENG (YandexIO)";
} // namespace

Device::Device(
    DeviceID deviceId,
    std::shared_ptr<Configuration> configuration,
    std::shared_ptr<ITelemetry> telemetry,
    std::unique_ptr<HAL> hal,
    std::optional<std::string> deviceColor,
    std::optional<std::string> platformRevision)
    : deviceId_{std::move(deviceId)}
    , configuration_{std::move(configuration)}
    , telemetry_{std::move(telemetry)}
    , hal_{std::move(hal)}
    , deviceColor_{std::move(deviceColor)}
    , platformRevision_{std::move(platformRevision)}
{
}

std::string Device::color() const {
    return deviceColor_.value_or("");
}

std::string Device::getUserAgent() const {
    constexpr std::string_view versionPlaceholder = "__QUASAR_VERSION_PLACEHOLDER__";
    std::string ua = quasar::tryGetString(configuration_->getCommonConfig(), "userAgent", DEFAULT_USER_AGENT);
    const auto placeholderPos = ua.find(versionPlaceholder);
    if (placeholderPos != std::string::npos) {
        ua.replace(placeholderPos, versionPlaceholder.length(), softwareVersion());
    }
    return ua;
}

std::string Device::getDeprecatedUserAgent() const {
    return configuration_->getDeviceType() + "/" + softwareVersion();
}

std::string Device::platformRevision() const {
    return platformRevision_.value_or("");
}

std::string Device::softwareVersion() const {
    return quasar::getString(configuration_->getCommonConfig(), "softwareVersion");
}
