#pragma once

#include <string>
#include <string_view>

namespace YandexIO {
    /**
     * @brief Device-specific asymmetric cryptography.
     *
     * A production device is likely to have a unique private key used for
     * authenticity checks. A secure implementation is likely to use a sealed
     * key storage (e.g. TrustZone), however a simple one might just store the
     * key in a file and process cryptography in-process.
     *
     * @note Cryptography implementation is assumed to be RSA, signature digest
     * is assumed to be sha256. These are implementation-defined, but should be
     * moved to interface specification.
     *
     * All methods throw exceptions in case of errors.
     */
    class DeviceCryptography {
    public:
        virtual ~DeviceCryptography();

        /**
         * @brief Sign an arbitrary data payload using the private key.
         *
         * Returns a base64-encoded signature payload.
         */
        virtual std::string sign(std::string_view data) = 0;

        /**
         * @brief Return a public key usable for RSA encryption.
         */
        virtual std::string getEncryptPublicKey() const = 0;

        /**
         * @brief Decrypt a data payload encrypted via \p getEncryptPublicKey().
         */
        virtual std::string decrypt(std::string_view data) = 0;

        virtual std::string getType() const = 0;
    };
} // namespace YandexIO
