#include "equalizer_config.h"

#include <yandex_io/libs/json_utils/json_utils.h>

#include <sstream>
#include <string>

using namespace quasar;

namespace YandexIO {

    namespace {

        std::string bandTypeToString(std::optional<EqualizerConfig::Band::Type> bandType) {
            if (!bandType.has_value()) {
                return "undefined";
            }

            switch (bandType.value()) {
                case EqualizerConfig::Band::Type::PEAK:
                    return "peak";
                case EqualizerConfig::Band::Type::LOW_SHELF:
                    return "low-shelf";
                case EqualizerConfig::Band::Type::HIGH_SHELF:
                    return "high-shelf";
            }
        }

    } // namespace

    Json::Value EqualizerConfig::toJson() const {
        Json::Value result;

        result["preventClipping"] = preventClipping;
        result["preset"] = preset;

        auto& bandsJson = result["bands"];
        for (const auto& band : bands) {
            Json::Value bandJson;
            bandJson["freq"] = band.freq;
            bandJson["width"] = band.width;
            bandJson["gain"] = band.gain;
            bandsJson.append(std::move(bandJson));
        }

        return result;
    }

    EqualizerConfig EqualizerConfig::fromJson(const Json::Value& config) {
        const Json::Value& bands = config["bands"];

        EqualizerConfig equalizerConfig;
        equalizerConfig.bands.reserve(bands.size());

        for (size_t i = 0; i < bands.size(); i++) {
            const auto& bandJson = bands.get(i, Json::Value());

            EqualizerConfig::Band band{
                .freq = bandJson["freq"].asDouble(),
                .width = bandJson["width"].asDouble(),
                .gain = bandJson["gain"].asDouble()};

            if (bandJson.isMember("type") && bandJson["type"].isString()) {
                const auto& type = bandJson["type"].asString();
                if (type == "peak") {
                    band.type = EqualizerConfig::Band::Type::PEAK;
                } else if (type == "low-shelf") {
                    band.type = EqualizerConfig::Band::Type::LOW_SHELF;
                } else if (type == "high-shelf") {
                    band.type = EqualizerConfig::Band::Type::HIGH_SHELF;
                }
            }

            equalizerConfig.bands.push_back(band);
        }

        equalizerConfig.preventClipping = config["preventClipping"].asBool();

        equalizerConfig.preset = tryGetString(config, "active_preset_id");

        return equalizerConfig;
    }

    EqualizerConfig EqualizerConfig::fromProto(const proto::EqualizerConfig& config) {
        EqualizerConfig equalizerConfig;
        equalizerConfig.bands.reserve(config.bands_size());

        for (int i = 0; i < config.bands_size(); i++) {
            const auto& bandProto = config.bands(i);

            EqualizerConfig::Band band{
                .freq = bandProto.freq(),
                .width = bandProto.width(),
                .gain = bandProto.gain()};

            if (bandProto.has_type()) {
                switch (bandProto.type()) {
                    case proto::EqualizerConfig::PEAK:
                        band.type = EqualizerConfig::Band::Type::PEAK;
                        break;
                    case proto::EqualizerConfig::LOW_SHELF:
                        band.type = EqualizerConfig::Band::Type::LOW_SHELF;
                        break;
                    case proto::EqualizerConfig::HIGH_SHELF:
                        band.type = EqualizerConfig::Band::Type::HIGH_SHELF;
                        break;
                    default:
                        break;
                }
            }

            equalizerConfig.bands.push_back(band);
        }

        equalizerConfig.preventClipping = config.prevent_clipping();

        return equalizerConfig;
    }

    std::ostream& operator<<(std::ostream& os, const EqualizerConfig& config) {
        os << "Equalizer config: preventClipping:" << std::to_string(config.preventClipping);

        for (const auto& band : config.bands) {
            os << " f:" << band.freq << " w:" << band.width << " g:" << band.gain << " t:" << bandTypeToString(band.type);
        }

        return os;
    }

} // namespace YandexIO
