#include "galois_field.h"

#include <iostream>
#include <stdexcept>
#include <string>

using namespace quasar;

GaloisField::GaloisField(int primitive, int size, int b)
{
    primitive_ = primitive;
    size_ = size;
    generatorBase_ = b;

    expTable_.resize(size_);
    logTable_.resize(size_);
    int x = 1;
    for (int i = 0; i < size_; i++)
    {
        expTable_[i] = x;
        x *= 2; // we're assuming the generator alpha is 2
        if (x >= size_)
        {
            x ^= primitive;
            x &= size_ - 1;
        }
    }
    for (int i = 0; i < size_ - 1; i++)
    {
        logTable_[expTable_[i]] = i;
    }
}

int GaloisField::addOrSubtract(int a, int b) const {
    if (a >= size_ || b >= size_) {
        throw std::runtime_error(
            "GaloisField::addOrSubtract(" + std::to_string(a) + "," + std::to_string(b) + ") "
                                                                                          "coefficient doesn't belong to the field");
    }
    return a ^ b;
}

int GaloisField::exp(int a) const {
    if (a >= size_) {
        throw std::runtime_error("GaloisField::exp(" + std::to_string(a) + ") coefficient doesn't belong to the field");
    }
    return expTable_[a];
}

int GaloisField::log(int a) const {
    if (a == 0 || a >= size_) {
        throw std::runtime_error(
            "GaloisField::log(" + std::to_string(a) + ") coefficient doesn't belong to the field or is zero");
    }
    return logTable_[a];
}

int GaloisField::inverse(int a) const {
    if (a == 0 || a >= size_) {
        throw std::runtime_error(
            "GaloisField::inverse(" + std::to_string(a) + ") coefficient doesn't belong to the field or is zero");
    }
    return expTable_[size_ - logTable_[a] - 1];
}

int GaloisField::multiply(int a, int b) const {
    if (a == 0 || b == 0) {
        return 0;
    }
    if (a >= size_ || b >= size_) {
        throw std::runtime_error(
            "GaloisField::multiply(" + std::to_string(a) + "," + std::to_string(b) + ") "
                                                                                     "coefficient doesn't belong to the field");
    }
    return expTable_[(logTable_[a] + logTable_[b]) % (size_ - 1)];
}

int GaloisField::getSize() const {
    return size_;
}

int GaloisField::getPrimitive() const {
    return primitive_;
}

int GaloisField::getGeneratorBase() const {
    return generatorBase_;
}

bool GaloisField::operator==(const GaloisField& other) const {
    return other.getSize() == size_ && other.getPrimitive() == primitive_ && other.getGeneratorBase() == generatorBase_;
}

bool GaloisField::operator!=(const GaloisField& other) const {
    return !((*this) == other);
}
