#pragma once

#include <vector>

/**
 * <p>This class contains utility methods for performing mathematical operations over
 * the Galois Fields. Operations use a given primitive polynomial in calculations.</p>
 *
 * <p>Port to C++ of Sean Owen's port to Java of C++ code.</p>
 *
 * @author William Rucklidge
 * @author Sean Owen
 * @author David Olivier
 * @author katayad (katayad@yandex-team.ru)
 */

namespace quasar {

    class GaloisField {
    public:
        GaloisField() = default;

        /*
         * primitive - irreducible polynomial whose coefficients are represented by the bits of an int,
         *             where the least-significant bit represents the constant coefficient
         * size      - the size of the field
         * b         - the factor b in the generator polynomial can be 0- or 1-based
         *             (g(x) = (x+a^b)(x+a^(b+1))...(x+a^(b+2t-1))). In most cases it should be 1, but for QR code it is 0.
         */
        GaloisField(int primitive, int size, int b);

        int addOrSubtract(int a, int b) const;

        int exp(int a) const;

        int log(int a) const;

        int inverse(int a) const;

        int multiply(int a, int b) const;

        int getSize() const;

        int getGeneratorBase() const;

        int getPrimitive() const;

        bool operator==(const GaloisField& other) const;

        bool operator!=(const GaloisField& other) const;

    private:
        int primitive_;
        int size_;
        int generatorBase_;
        std::vector<int> expTable_;
        std::vector<int> logTable_;
    };

} // namespace quasar
