#include "galois_field_polynomial.h"

#include "galois_field.h"

#include <iostream>
#include <stdexcept>
#include <string>

using namespace quasar;

GaloisFieldPolynomial::GaloisFieldPolynomial(GaloisField field, const std::vector<int>& coefficients)
    : field_(field)
    , coefficients_(coefficients)
{
    for (int coef : coefficients) {
        if (coef >= field_.getSize()) {
            throw std::runtime_error(
                "GaloisFieldPolynomial() coefficient doesn't belong to the field_: " + std::to_string(coef));
        }
    }
    if (coefficients.size() == 0) {
        throw std::runtime_error("GaloisFieldPolynomial() no coefficients");
    }
    int coefficientsLength = coefficients.size();
    if (coefficientsLength > 1 && coefficients[0] == 0)
    {
        // Leading term must be non-zero for anything except the constant polynomial "0"
        int firstNonZero = 1;
        while (firstNonZero < coefficientsLength && coefficients[firstNonZero] == 0)
        {
            firstNonZero++;
        }
        if (firstNonZero == coefficientsLength)
        {
            coefficients_ = std::vector<int>{0};
        } else {
            coefficients_ = std::vector<int>(coefficients.begin() + firstNonZero, coefficients.end());
        }
    } else {
        coefficients_ = coefficients;
    }
}

std::vector<int> GaloisFieldPolynomial::getCoefficients() const {
    return coefficients_;
}

int GaloisFieldPolynomial::getDegree() const {
    return coefficients_.size() - 1;
}

bool GaloisFieldPolynomial::isZero() const {
    return coefficients_[0] == 0;
}

int GaloisFieldPolynomial::getCoefficient(int degree) const {
    return coefficients_[coefficients_.size() - 1 - degree];
}

int GaloisFieldPolynomial::evaluateAt(int a) const {
    if (a == 0)
    {
        // Just return the x^0 coefficient
        return getCoefficient(0);
    }
    size_t size = coefficients_.size();
    if (a == 1)
    {
        // Just the sum of the coefficients
        int result = 0;
        for (int coefficient : coefficients_)
        {
            result = field_.addOrSubtract(result, coefficient);
        }
        return result;
    }
    int result = coefficients_[0];
    for (size_t i = 1; i < size; i++)
    {
        result = field_.addOrSubtract(field_.multiply(a, result), coefficients_[i]);
    }
    return result;
}

GaloisFieldPolynomial GaloisFieldPolynomial::addOrSubtract(const GaloisFieldPolynomial& other)
{
    if (field_ != other.getField())
    {
        throw std::runtime_error("GenericGFPolys do not have same GenericGF field");
    }
    if (isZero())
    {
        return other;
    }
    if (other.isZero())
    {
        return (*this);
    }

    std::vector<int> smallerCoefficients = coefficients_;
    std::vector<int> largerCoefficients = other.getCoefficients();
    if (smallerCoefficients.size() > largerCoefficients.size())
    {
        std::vector<int> temp = smallerCoefficients;
        smallerCoefficients = largerCoefficients;
        largerCoefficients = temp;
    }
    std::vector<int> sumDiff(largerCoefficients.size());
    int lengthDiff = largerCoefficients.size() - smallerCoefficients.size();
    // Copy high-order terms only found in higher-degree polynomial's coefficients
    sumDiff = std::vector<int>(largerCoefficients.begin(), largerCoefficients.end());

    for (uint i = lengthDiff; i < largerCoefficients.size(); i++)
    {
        sumDiff[i] = field_.addOrSubtract(smallerCoefficients[i - lengthDiff], largerCoefficients[i]);
    }
    return GaloisFieldPolynomial(field_, sumDiff);
}

GaloisField GaloisFieldPolynomial::getField() const {
    return field_;
}

GaloisFieldPolynomial GaloisFieldPolynomial::multiply(const GaloisFieldPolynomial& other)
{
    if (field_ != other.getField())
    {
        throw std::runtime_error("GenericGFPolys do not have same GenericGF field");
    }
    if (isZero() || other.isZero())
    {
        return GaloisFieldPolynomial(field_, std::vector<int>{0});
    }
    std::vector<int> aCoefficients = coefficients_;
    int aLength = aCoefficients.size();
    std::vector<int> bCoefficients = other.getCoefficients();
    int bLength = bCoefficients.size();
    std::vector<int> product(aLength + bLength - 1, 0);
    for (int i = 0; i < aLength; i++)
    {
        int aCoeff = aCoefficients[i];
        for (int j = 0; j < bLength; j++)
        {
            product[i + j] = field_.addOrSubtract(product[i + j],
                                                  field_.multiply(aCoeff, bCoefficients[j]));
        }
    }
    return GaloisFieldPolynomial(field_, product);
}

GaloisFieldPolynomial GaloisFieldPolynomial::multiply(int scalar)
{
    if (scalar == 0)
    {
        return GaloisFieldPolynomial(field_, std::vector<int>{0});
    }
    if (scalar == 1)
    {
        return (*this);
    }
    int size = coefficients_.size();
    std::vector<int> product(size, 0);
    for (int i = 0; i < size; i++)
    {
        product[i] = field_.multiply(coefficients_[i], scalar);
    }
    return GaloisFieldPolynomial(field_, product);
}

GaloisFieldPolynomial GaloisFieldPolynomial::multiplyByMonomial(int degree, int coefficient)
{
    if (degree < 0)
    {
        throw std::runtime_error("GenericGFPolys do not have same GenericGF field");
    }
    if (coefficient == 0)
    {
        return GaloisFieldPolynomial(field_, std::vector<int>{0});
    }
    int size = coefficients_.size();
    std::vector<int> product(size + degree, 0);
    for (int i = 0; i < size; i++)
    {
        product[i] = field_.multiply(coefficients_[i], coefficient);
    }
    return GaloisFieldPolynomial(field_, product);
}

std::pair<GaloisFieldPolynomial, GaloisFieldPolynomial> GaloisFieldPolynomial::divide(GaloisFieldPolynomial other)
{
    if (field_ != other.getField())
    {
        throw std::runtime_error("GenericGFPolys do not have same GenericGF field");
    }
    if (other.isZero())
    {
        throw std::runtime_error("Divide by 0");
    }

    GaloisFieldPolynomial quotient = GaloisFieldPolynomial(field_, std::vector<int>{0});
    GaloisFieldPolynomial remainder = (*this);

    int denominatorLeadingTerm = other.getCoefficient(other.getDegree());
    int inverseDenominatorLeadingTerm = field_.inverse(denominatorLeadingTerm);

    while (remainder.getDegree() >= other.getDegree() && !remainder.isZero())
    {
        int degreeDifference = remainder.getDegree() - other.getDegree();
        int scale = field_.multiply(remainder.getCoefficient(remainder.getDegree()), inverseDenominatorLeadingTerm);
        GaloisFieldPolynomial term = other.multiplyByMonomial(degreeDifference, scale);
        GaloisFieldPolynomial iterationQuotient = buildMonomial(field_, degreeDifference, scale);
        quotient = quotient.addOrSubtract(iterationQuotient);
        remainder = remainder.addOrSubtract(term);
    }

    return std::make_pair(quotient, remainder);
}

GaloisFieldPolynomial GaloisFieldPolynomial::buildMonomial(const GaloisField& field, int degree, int coefficient)
{
    if (degree < 0)
    {
        throw std::runtime_error("ARG");
    }
    if (coefficient == 0)
    {
        return GaloisFieldPolynomial(field, std::vector<int>{0});
    }
    std::vector<int> coefficients(degree + 1, 0);
    coefficients[0] = coefficient;
    return GaloisFieldPolynomial(field, coefficients);
}

std::string GaloisFieldPolynomial::toString()
{
    std::string result;
    for (int degree = getDegree(); degree >= 0; degree--)
    {
        int coefficient = getCoefficient(degree);
        if (coefficient != 0)
        {
            if (coefficient < 0)
            {
                result += " - ";
                coefficient = -coefficient;
            } else {
                if (result.length() > 0)
                {
                    result += " + ";
                }
            }
            if (degree == 0 || coefficient != 1)
            {
                int alphaPower = field_.log(coefficient);
                if (alphaPower == 0)
                {
                    result += '1';
                } else if (alphaPower == 1)
                {
                    result += "";
                } else {
                    result += "";
                    result += std::to_string(alphaPower);
                }
            }
            if (degree != 0)
            {
                if (degree == 1)
                {
                    result += 'x';
                } else {
                    result += "x^";
                    result += std::to_string(degree);
                }
            }
        }
    }
    return result;
}
