#include "reed_solomon_encoder.h"

#include "galois_field.h"

#include <iostream>
#include <stdexcept>
#include <string>

using namespace quasar;

ReedSolomonEncoder::ReedSolomonEncoder(const GaloisField& field)
{
    field_ = field;
    cachedGenerators_ = std::vector<GaloisFieldPolynomial>();
    cachedGenerators_.push_back(GaloisFieldPolynomial(field_, std::vector<int>{1}));
}

GaloisFieldPolynomial ReedSolomonEncoder::buildGenerator(uint degree) const {
    if (degree >= cachedGenerators_.size())
    {
        GaloisFieldPolynomial lastGenerator = cachedGenerators_[cachedGenerators_.size() - 1];
        for (uint d = cachedGenerators_.size(); d <= degree; d++)
        {
            GaloisFieldPolynomial nextGenerator = lastGenerator.multiply(
                GaloisFieldPolynomial(field_, std::vector<int>{1, field_.exp(d - 1 + field_.getGeneratorBase())}));
            cachedGenerators_.push_back(nextGenerator);
            lastGenerator = nextGenerator;
        }
    }
    return cachedGenerators_[degree];
}

void ReedSolomonEncoder::encode(std::vector<int>& toEncode, int ecBytes) const {
    if (ecBytes == 0)
    {
        throw std::runtime_error("No error correction bytes");
    }
    int dataBytes = toEncode.size() - ecBytes;
    if (dataBytes <= 0)
    {
        throw std::runtime_error("No data bytes provided: " + std::to_string(dataBytes));
    }
    GaloisFieldPolynomial generator = buildGenerator(ecBytes);
    std::vector<int> infoCoefficients(toEncode.begin(), toEncode.begin() + dataBytes);
    GaloisFieldPolynomial info = GaloisFieldPolynomial(field_, infoCoefficients);
    info = info.multiplyByMonomial(ecBytes, 1);
    GaloisFieldPolynomial quot = info.divide(generator).first;

    GaloisFieldPolynomial remainder = info.divide(generator).second;

    std::vector<int> coefficients_ = remainder.getCoefficients();
    int numZeroCoefficients = ecBytes - coefficients_.size();
    for (int i = 0; i < numZeroCoefficients; i++)
    {
        toEncode[dataBytes + i] = 0;
    }
    for (uint i = 0; i < coefficients_.size(); ++i)
    {
        toEncode[dataBytes + numZeroCoefficients + i] = coefficients_[i];
    }
}
