#include "account_devices.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <algorithm>
#include <chrono>
#include <random>

YIO_DEFINE_LOG_MODULE("glagol_sdk");

namespace glagol {
    namespace {

        constexpr std::chrono::seconds DEFAULT_FAIL_REQUEST_INTERVAL{60};
        constexpr std::chrono::seconds BURST_FAIL_REQUEST_INTERVAL{15};
        constexpr std::chrono::seconds MAXIMUM_FAIL_REQUEST_INTERVAL{5 * DEFAULT_FAIL_REQUEST_INTERVAL};
        constexpr std::chrono::seconds DEFAULT_SUCCESS_REQUEST_INTERVAL{600};

        const auto PROCESS_START_TIME = std::chrono::steady_clock::now();
        constexpr std::chrono::seconds BURST_START_INTERVAL{60};
        constexpr std::chrono::seconds UPDATE_DEVICES_DELAY{30};

        std::chrono::seconds failRequestInterval(int failCounter)
        {
            static std::mt19937 randomGenerator((std::random_device())());

            if (failCounter <= 1 || std::chrono::steady_clock::now() - PROCESS_START_TIME < BURST_START_INTERVAL) {
                std::uniform_int_distribution<long> distribution{0, BURST_FAIL_REQUEST_INTERVAL.count() / 2};
                return BURST_FAIL_REQUEST_INTERVAL + std::chrono::seconds(distribution(randomGenerator));
            } else {
                std::uniform_int_distribution<long> distribution{0, DEFAULT_FAIL_REQUEST_INTERVAL.count() / 2};
                return std::min(failCounter * DEFAULT_FAIL_REQUEST_INTERVAL, MAXIMUM_FAIL_REQUEST_INTERVAL) + std::chrono::seconds(distribution(randomGenerator));
            }
        }

    } // namespace

    AccountDevices::AccountDevices(std::shared_ptr<BackendApi> backendApi,
                                   std::string accountDeviceCacheFile,
                                   bool updateSuspended)
        : backendApi_(std::move(backendApi))
        , accountDeviceCacheFile_(std::move(accountDeviceCacheFile))
        , backendApiQueue_(std::make_shared<quasar::NamedCallbackQueue>("AccountDevices"))
        , deviceListChangedSignal_(
              [this](bool /*onConnect*/) {
                  std::lock_guard<std::mutex> lock(mutex_);
                  return std::make_tuple(accountDevices_);
              }, serviceLifetime_)
        , lastTryTime_(std::chrono::steady_clock::now() - UPDATE_DEVICES_DELAY)
        , backendSuspended_(updateSuspended || !backendApi_->settings())
        , updateAccountDeviceList_(backendApiQueue_, quasar::UniqueCallback::ReplaceType::INSERT_BACK)
    {
        loadAccountDeviceCacheUnsafe();

        if (!backendSuspended_) {
            updateAccountDeviceList_.execute([this] { updateAccountDeviceList(0); }, serviceLifetime_);
        }
    }

    AccountDevices::~AccountDevices()
    {
        serviceLifetime_.die();
    }

    bool AccountDevices::setSettings(const BackendSettings& settings)
    {
        if (!backendApi_->setSettings(settings)) {
            return false;
        }

        std::lock_guard<std::mutex> lock(mutex_);
        updateAccountDeviceList_.reset();
        backendSuspended_ = backendSuspended_ || !settings;
        if (!backendSuspended_) {
            updateAccountDeviceList_.execute([this] { updateAccountDeviceList(0); }, serviceLifetime_);
        }
        return true;
    }

    void AccountDevices::suspendUpdate() noexcept {
        std::lock_guard<std::mutex> lock(mutex_);
        updateAccountDeviceList_.reset();
        backendSuspended_ = true;
    }

    bool AccountDevices::resumeUpdate() noexcept {
        std::lock_guard<std::mutex> lock(mutex_);
        auto settings = backendApi_->settings();
        if (!settings) {
            return false;
        }
        if (backendSuspended_) {
            backendSuspended_ = false;
            updateAccountDeviceList_.execute([this] { updateAccountDeviceList(0); }, serviceLifetime_);
        }
        return true;
    }

    bool AccountDevices::scheduleUpdate() noexcept {
        std::lock_guard<std::mutex> lock(mutex_);
        const auto sinceLastTry = std::chrono::duration_cast<std::chrono::seconds>(std::chrono::steady_clock::now() - lastTryTime_);
        auto settings = backendApi_->settings();
        if (!settings) {
            return false;
        }
        updateAccountDeviceList_.reset();
        if (sinceLastTry >= UPDATE_DEVICES_DELAY) {
            updateAccountDeviceList_.execute([this] { updateAccountDeviceList(0); }, serviceLifetime_);
        } else {
            updateAccountDeviceList_.executeDelayed([this] { updateAccountDeviceList(0); }, UPDATE_DEVICES_DELAY - sinceLastTry, serviceLifetime_);
        }
        return true;
    }

    IBackendApi::DevicesMap AccountDevices::devices() const noexcept {
        std::lock_guard<std::mutex> lock(mutex_);
        return accountDevices_;
    }

    AccountDevices::IDeviceListChangedSignal& AccountDevices::deviceListChangedSignal() noexcept {
        return deviceListChangedSignal_;
    }

    void AccountDevices::updateAccountDeviceList(int failCounter) noexcept {
        std::chrono::seconds nextTryTimeout = DEFAULT_SUCCESS_REQUEST_INTERVAL;
        IBackendApi::DevicesMap deviceMap;
        bool failed = false;
        try {
            lastTryTime_ = std::chrono::steady_clock::now();
            deviceMap = backendApi_->getConnectedDevicesList();
        } catch (const std::exception& ex) {
            YIO_LOG_ERROR_EVENT("AccountDevices.GetConnectedDevicesListFailed", "Fail to update account device list: " << ex.what());
            failed = true;
        }

        std::unique_lock<std::mutex> lock(mutex_);
        bool deviceChanged = false;
        if (failed) {
            ++failCounter;
            nextTryTimeout = failRequestInterval(failCounter);
        } else {
            if (deviceMap != accountDevices_) {
                accountDevices_ = deviceMap;
                saveAccountDeviceCacheUnsafe();
                deviceChanged = true;
                YIO_LOG_DEBUG("List of account devices:");
                for (const auto& [deviceId, device] : accountDevices_) {
                    YIO_LOG_DEBUG("    deviceId=" << deviceId.id << ", platform=" << deviceId.platform << ", name=" << device.name);
                }
            }
            failCounter = 0;
            nextTryTimeout = DEFAULT_SUCCESS_REQUEST_INTERVAL;
        }
        YIO_LOG_DEBUG("Schedule next try to update account device list: timeout=" << nextTryTimeout.count() << " seconds");
        updateAccountDeviceList_.executeDelayed([this, failCounter] { updateAccountDeviceList(failCounter); }, nextTryTimeout, serviceLifetime_);
        lock.unlock();

        if (deviceChanged) {
            YIO_LOG_DEBUG("Notify all about changes");
            deviceListChangedSignal_.emit();
        }
    }

    void AccountDevices::loadAccountDeviceCacheUnsafe() noexcept {
        if (accountDeviceCacheFile_.empty()) {
            return;
        }
        try {
            YIO_LOG_DEBUG("Load account devices from cache");
            auto jsonCache = quasar::getConfigFromFile(accountDeviceCacheFile_);
            auto jsonDevices = quasar::tryGetArray(jsonCache, "devices");
            for (const auto& jsonDevice : jsonDevices) {
                auto idDevicePair = IBackendApi::fromJson(jsonDevice);
                if (!idDevicePair.first) {
                    continue;
                }
                accountDevices_.emplace(std::move(idDevicePair));
            }
        } catch (...) {
        }
    }

    void AccountDevices::saveAccountDeviceCacheUnsafe() noexcept {
        if (accountDeviceCacheFile_.empty()) {
            return;
        }
        try {
            YIO_LOG_DEBUG("Save my device to cache " << accountDeviceCacheFile_);
            Json::Value jsonCache;
            Json::Value& jsonDevices = jsonCache["devices"];
            jsonDevices = Json::Value{Json::arrayValue};
            for (const auto& [deviceId, device] : accountDevices_) {
                auto jDevice = device.serialize(deviceId);
                jDevice.removeMember("glagol");
                jsonDevices.append(jDevice);
            }
            quasar::TransactionFile file(accountDeviceCacheFile_);
            file.write(quasar::jsonToString(jsonCache));
            file.commit();
        } catch (const std::exception& ex) {
            YIO_LOG_DEBUG("Fail to save devices to cache: " << ex.what());
        }
    }

} // namespace glagol
