/*
 * A lot of code was copied from https://github.com/lathiat/avahi/blob/master/examples/core-browse-services.c
 */
#include "avahi_browse_client.h"

#include <yandex_io/libs/glagol_sdk/quasar_includes.h>

#include <yandex_io/libs/base/named_callback_queue.h>

#include <algorithm>
#include <cstdio>
#include <cstring>
#include <functional>
#include <iostream>

YIO_DEFINE_LOG_MODULE("glagol_sdk");

using namespace glagol;

AvahiBrowseClient::Item::Info glagol::makeInfo(const char* hostname, int port, const char* address, AvahiStringList* txtList)
{
    std::map<std::string, std::string> txt;
    /*
     * based on avahi_string_list_to_string in avahi/avahi-common/strlst.c
     */
    std::string entry;
    for (auto n = txtList; n; n = n->next) {
        entry.assign((char*)n->text, n->size);
        auto pos = entry.find('=');
        if (pos == std::string::npos) {
            continue;
        }
        txt[entry.substr(0, pos)] = entry.substr(pos + 1, entry.size() - pos - 1);
    }
    return AvahiBrowseClient::Item::Info{hostname, port, address, std::move(txt)};
}

AvahiBrowseClient::InitSate AvahiBrowseClient::initState_ = AvahiBrowseClient::InitSate::NOT_INITED;

AvahiBrowseClient::AvahiBrowseClient(std::shared_ptr<quasar::ICallbackQueue> callbackQueue, const quasar::Lifetime& lifetime, bool ignoreIpv6,
                                     std::optional<std::string> onlyNameToResolve)
    : ignoreIpv6_(ignoreIpv6)
    , onlyNameToResolve_(std::move(onlyNameToResolve))
    , callbackQueue_(std::move(callbackQueue))
    , resolvedItemsChangedSignal_(
          [this](bool onConnect) -> std::tuple<ResolvedItems> {
              if (onConnect && !isBrowsing()) {
                  startBrowsing();
              }
              return std::make_tuple(resolved_items_);
          }, lifetime)
{
    if (!callbackQueue_) {
        throw std::invalid_argument("Fail to create AvahiBrowseClient with null callback queue");
    }
    if (onlyNameToResolve_) {
        YIO_LOG_INFO("AvahiBrowseClient only for " << *onlyNameToResolve_);
    }
}

AvahiBrowseClient::AvahiBrowseClient(std::optional<std::string> onlyNameToResolve)
    : AvahiBrowseClient(std::make_shared<quasar::NamedCallbackQueue>("AvahiBrowseClient"), quasar::Lifetime::immortal, false, std::move(onlyNameToResolve))
{
}

AvahiBrowseClient::~AvahiBrowseClient()
{
    stopBrowsing();
    callbackQueue_.reset(); // finish queue before any class member will be destroyed
}

void AvahiBrowseClient::resolveCallbackHelper(AvahiSServiceResolver* resolver, AvahiIfIndex /* interface */, AvahiProtocol /* protocol */,
                                              AvahiResolverEvent event, const char* name, const char* type,
                                              const char* domain, const char* host_name, const AvahiAddress* address,
                                              uint16_t port, AvahiStringList* txt, AvahiLookupResultFlags /* flags */,
                                              void* userdata) {
    Y_VERIFY(resolver);
    auto obj = reinterpret_cast<AvahiBrowseClient*>(userdata);
    obj->resolveCallback(event, name, type, domain, host_name, address, port, txt);
    avahi_s_service_resolver_free(resolver);
}

void AvahiBrowseClient::browseCallbackHelper(AvahiSServiceBrowser* browser [[maybe_unused]], AvahiIfIndex ifIndex, AvahiProtocol protocol,
                                             AvahiBrowserEvent event, const char* name, const char* type,
                                             const char* domain, AvahiLookupResultFlags /* flags */, void* userdata) {
    Y_VERIFY(browser);
    auto obj = reinterpret_cast<AvahiBrowseClient*>(userdata);
    obj->browseCallback(ifIndex, protocol, event, name, type, domain);
}

void AvahiBrowseClient::resolveCallback(AvahiResolverEvent event, const char* name, const char* type,
                                        const char* domain, const char* host_name, const AvahiAddress* address,
                                        uint16_t port, AvahiStringList* txt) {
    /* Called whenever a service has been resolved successfully or timed out */

    switch (event) {
        case AVAHI_RESOLVER_FAILURE: {
            auto item = Item({std::string(name), std::string(type), std::string(domain)});
            auto error = Error({avahi_server_errno(server_), avahi_strerror(avahi_server_errno(server_))});
            callbackQueue_->add([this, item, error]() { onResolveFailure(item, error); });
            break;
        }
        case AVAHI_RESOLVER_FOUND: {
            char a[AVAHI_ADDRESS_STR_MAX];
            avahi_address_snprint(a, sizeof(a), address);

            auto item = Item({std::string(name), std::string(type), std::string(domain)});
            auto info = makeInfo(host_name, port, a, txt);
            callbackQueue_->add([this, item, info]() { onResolved(item, info); });
            break;
        }
    }
}

void AvahiBrowseClient::browseCallback(AvahiIfIndex ifIndex, AvahiProtocol protocol, AvahiBrowserEvent event, const char* name, const char* type, const char* domain) {
    /* Called whenever a new services becomes available on the LAN or is removed from the LAN */

    switch (event) {
        case AVAHI_BROWSER_FAILURE: {
            stopBrowsing();
            auto error = Error({avahi_server_errno(server_), avahi_strerror(avahi_server_errno(server_))});
            callbackQueue_->add([this, error]() { onBrowserFailure(error); });
            break;
        }
        case AVAHI_BROWSER_NEW: {
            /* We ignore the returned resolver object. In the callback
             function we free it. If the server is terminated before
             the callback function is called the server will free
             the resolver for us. */

            std::string strName(name);

            if (!onlyNameToResolve_ || strName.find(*onlyNameToResolve_) != std::string::npos) {
                auto item = Item({std::move(strName), std::string(type), std::string(domain)});
                if (!(avahi_s_service_resolver_new(server_, ifIndex, protocol, name, type, domain,
                                                   AVAHI_PROTO_UNSPEC, (AvahiLookupFlags)0, resolveCallbackHelper, this))) {
                    auto error = Error({avahi_server_errno(server_), avahi_strerror(avahi_server_errno(server_))});
                    callbackQueue_->add([this, item, error]() { onResolveFailure(item, error); });
                }
                callbackQueue_->add([this, item]() { onBrowserNew(item); });
            }
            break;
        }

        case AVAHI_BROWSER_REMOVE: {
            Item item{std::string(name), std::string(type), std::string(domain)};
            callbackQueue_->add([this, item]() { onBrowserRemove(item); });
            break;
        }

        case AVAHI_BROWSER_ALL_FOR_NOW:
            break;
        case AVAHI_BROWSER_CACHE_EXHAUSTED:
            break;
    }
}

void AvahiBrowseClient::startAvahiPollLoop() {
    AvahiServerConfig config;
    int error;

    /* Allocate main loop object */
    if (!(simple_poll_ = avahi_simple_poll_new())) {
        clean();
        throw Exception("Failed to create simple poll object.");
    }

    /* Do not publish any local records */
    avahi_server_config_init(&config);
    config.use_ipv6 = 0;
    config.publish_hinfo = 0;
    config.publish_addresses = 0;
    config.publish_workstation = 0;
    config.publish_domain = 0;
    config.disable_publishing = 1;
    config.deny_interfaces = avahi_string_list_new("lo", nullptr);

    /* Allocate a new server */
    server_ = avahi_server_new(avahi_simple_poll_get(simple_poll_), &config, nullptr, nullptr, &error);

    /* Free the configuration data */
    avahi_server_config_free(&config);

    /* Check whether creating the server object succeeded */
    if (!server_) {
        clean();
        throw Exception("Failed to create server: %s\n" + std::string(avahi_strerror(error)));
    }

    /* Create the service browser */
    if (!(sb_ = avahi_s_service_browser_new(
              server_, AVAHI_IF_UNSPEC, AVAHI_PROTO_UNSPEC, serviceType_.c_str(), nullptr, (AvahiLookupFlags)0, browseCallbackHelper, this))) {
        clean();
        throw Exception("Failed to create service browser: " + std::string(avahi_strerror(avahi_server_errno(server_))));
    }

    /* Run the main loop */
    avahi_simple_poll_loop(simple_poll_);

    /* Clean up all */
    clean();
}

void AvahiBrowseClient::startBrowsing() {
    std::unique_lock<std::mutex> lock(mutex_);
    if (initState_ == InitSate::INITED_FOR_TEST) {
        YIO_LOG_INFO("Test Mode. Ignoring startBrowsing");
        return;
    }
    if (isBrowsing_) {
        YIO_LOG_ERROR_EVENT("AvahiBrowseClient.StartBrowsingWhileBrowsing", "Already browsing");
        return;
    }
    isBrowsing_ = true;
    avahiServerThread_ = std::thread(&AvahiBrowseClient::startAvahiPollLoop, this);
}

void AvahiBrowseClient::stopBrowsing() {
    std::unique_lock<std::mutex> lock(mutex_);
    if (initState_ == InitSate::INITED_FOR_TEST) {
        YIO_LOG_INFO("Test Mode. Ignoring stopBrowsing");
        return;
    }
    if (!isBrowsing_) {
        YIO_LOG_ERROR_EVENT("AvahiBrowseClient.StopBrowsingWhileNotBrowsing", "Not browsing");
        return;
    }
    isBrowsing_ = false;
    avahi_simple_poll_quit(simple_poll_);
    if (avahiServerThread_.joinable()) {
        avahiServerThread_.join();
    }
}

bool AvahiBrowseClient::isBrowsing() {
    std::unique_lock<std::mutex> lock(mutex_);
    return isBrowsing_;
}

void AvahiBrowseClient::clean() {
    if (sb_) {
        avahi_s_service_browser_free(sb_);
        sb_ = nullptr;
    }

    if (server_) {
        avahi_server_free(server_);
        server_ = nullptr;
    }

    if (simple_poll_) {
        avahi_simple_poll_free(simple_poll_);
        simple_poll_ = nullptr;
    }
}

void AvahiBrowseClient::getResolvedItems(
    std::function<void(const std::unordered_map<Item, Item::Info, Item::Hasher>&)> resolvedItemsHandler) {
    callbackQueue_->add([this, resolvedItemsHandler]() { resolvedItemsHandler(this->resolved_items_); });
}

void AvahiBrowseClient::initForTest() {
    if (initState_ == InitSate::INITED) {
        throw std::runtime_error("Double Init");
    }
    initState_ = InitSate::INITED_FOR_TEST;
}

AvahiBrowseClient::TestEventTrigger AvahiBrowseClient::getTestEventTrigger() {
    if (initState_ != InitSate::INITED_FOR_TEST) {
        throw std::runtime_error("Not in test mode");
    }
    return TestEventTrigger{this};
}

AvahiBrowseClient::IResolvedItemsChangedSignal& AvahiBrowseClient::resolvedItemsChangedSignal()
{
    return resolvedItemsChangedSignal_;
}

void AvahiBrowseClient::addListener(std::weak_ptr<Listener> listener) {
    std::lock_guard<std::mutex> lock(listenersMutex_);
    if (listeners_.end() == std::find_if(listeners_.begin(), listeners_.end(),
                                         [listener](std::weak_ptr<Listener> wp) {
                                             return !(wp.owner_before(listener) || listener.owner_before(wp));
                                         }))
    {
        listeners_.push_back(listener);
        if (!isBrowsing()) {
            startBrowsing();
        }
    }
}

void AvahiBrowseClient::removeListener(std::weak_ptr<Listener> listener) {
    std::lock_guard<std::mutex> lock(listenersMutex_);

    listeners_.remove_if([listener](std::weak_ptr<Listener> wp) {
        return !(wp.owner_before(listener) || listener.owner_before(wp));
    });

    if (listeners_.empty() && isBrowsing()) {
        stopBrowsing();
    }
}

std::list<std::weak_ptr<AvahiBrowseClient::Listener>> AvahiBrowseClient::copyListeners() {
    std::lock_guard lock(listenersMutex_);

    auto listenersCopy = listeners_;
    return listenersCopy;
}

void AvahiBrowseClient::onBrowserFailure(const Error& error) {
    auto listenersCopy = copyListeners();
    for (auto& listener : listenersCopy) {
        auto listener_lock = listener.lock();
        if (listener_lock) {
            listener_lock->onBrowserFailure(error);
        }
    }
}

void AvahiBrowseClient::onBrowserNew(const AvahiBrowseClient::Item& item) {
    YIO_LOG_DEBUG("Avahi: browse new: name=" << item.name << ", type=" << item.type << ", domain=" << item.domain);
    auto listenersCopy = copyListeners();
    for (auto& listener : listenersCopy) {
        auto listener_lock = listener.lock();
        if (listener_lock) {
            listener_lock->onBrowserNew(item);
        }
    }
}

void AvahiBrowseClient::onBrowserRemove(const AvahiBrowseClient::Item& item) {
    YIO_LOG_DEBUG("Avahi: browse remove: name=" << item.name << ", type=" << item.type << ", domain=" << item.domain);
    auto listenersCopy = copyListeners();
    for (auto& listener : listenersCopy) {
        auto listener_lock = listener.lock();
        if (listener_lock) {
            listener_lock->onBrowserRemove(item);
        }
    }
    if (resolved_items_.count(item)) {
        auto info = resolved_items_[item];
        for (auto& listener : listenersCopy) {
            auto listener_lock = listener.lock();
            if (listener_lock) {
                listener_lock->onResolvedRemoved(item, info);
            }
        }
        resolved_items_.erase(item);
    }
    resolvedItemsChangedSignal_.emit();
}

void AvahiBrowseClient::onResolveFailure(const AvahiBrowseClient::Item& item, const Error& error) {
    auto listenersCopy = copyListeners();
    for (auto& listener : listenersCopy) {
        auto listener_lock = listener.lock();
        if (listener_lock) {
            listener_lock->onResolveFailure(item, error);
        }
    }
}

void AvahiBrowseClient::onResolved(const AvahiBrowseClient::Item& item, const Item::Info& info) {
    YIO_LOG_DEBUG("Avahi: resolved: name=" << item.name << ", type=" << item.type << ", domain=" << item.domain << ", hostname=" << info.hostname << ", port=" << info.port << ", address=" << info.address);
    if (ignoreIpv6_ && info.address.find(':') != std::string::npos) {
        return;
    }
    if (info.address.find("169.254.") == 0) {
        // Used for link-local addresses[6] between two hosts on a single link when no IP address is otherwise specified, such as would have normally been retrieved from a DHCP server.
        return;
    }
    resolved_items_[item] = info;
    auto listenersCopy = copyListeners();
    for (auto& listener : listenersCopy) {
        auto listener_lock = listener.lock();
        if (listener_lock) {
            listener_lock->onNewResolved(item, info);
        }
    }
    resolvedItemsChangedSignal_.emit();
}

void AvahiBrowseClient::TestEventTrigger::onBrowserFailure(const AvahiBrowseClient::Error& error) const {
    client->callbackQueue_->add([client = client, error]() {
        client->onBrowserFailure(error);
    });
}

void AvahiBrowseClient::TestEventTrigger::onBrowserNew(const AvahiBrowseClient::Item& item) const {
    client->callbackQueue_->add([client = client, item]() {
        client->onBrowserNew(item);
    });
}

void AvahiBrowseClient::TestEventTrigger::onBrowserRemove(const AvahiBrowseClient::Item& item) const {
    client->callbackQueue_->add([client = client, item]() {
        client->onBrowserRemove(item);
    });
}

void AvahiBrowseClient::TestEventTrigger::onResolveFailure(const AvahiBrowseClient::Item& item,
                                                           const AvahiBrowseClient::Error& error) const {
    client->callbackQueue_->add([client = client, item, error]() {
        client->onResolveFailure(item, error);
    });
}

void AvahiBrowseClient::TestEventTrigger::onNewResolved(const AvahiBrowseClient::Item& item,
                                                        const AvahiBrowseClient::Item::Info& info) const {
    client->callbackQueue_->add([client = client, item, info]() {
        client->onResolved(item, info);
    });
}
