#pragma once

#include "i_avahi_browse_client.h"

#include <yandex_io/libs/glagol_sdk/quasar_includes.h>

#include <yandex_io/libs/signals/signal_external.h>
#include <yandex_io/libs/threading/lifetime.h>

#include <avahi-common/error.h>
#include <avahi-common/malloc.h>
#include <avahi-common/simple-watch.h>

#include <avahi-core/core.h>
#include <avahi-core/lookup.h>

#include <functional>
#include <list>
#include <map>
#include <mutex>
#include <optional>
#include <thread>
#include <unordered_map>

namespace glagol {

    class AvahiBrowseClient: public IAvahiBrowseClient {
    public:
        class Exception: public std::runtime_error {
        public:
            explicit Exception(const std::string& message)
                : std::runtime_error(message) {
            }
            virtual ~Exception() = default;
        };
        struct Listener {
            virtual ~Listener() = default;

            virtual void onBrowserFailure(const Error& error) = 0;
            virtual void onBrowserNew(const Item& item) = 0;
            virtual void onBrowserRemove(const Item& item) = 0;
            virtual void onResolveFailure(const Item& item, const Error& error) = 0;
            virtual void onNewResolved(const Item& item, const Item::Info& info) = 0;
            virtual void onResolvedRemoved(const Item& item, const Item::Info& info) = 0;
        };

        AvahiBrowseClient(std::shared_ptr<quasar::ICallbackQueue> callbackQueue,
                          const quasar::Lifetime& lifetime,
                          bool ignoreIpv6 = false,
                          std::optional<std::string> onlyNameToResolve = std::nullopt);

        AvahiBrowseClient(std::optional<std::string> onlyNameToResolve = std::nullopt);
        ~AvahiBrowseClient();

        void getResolvedItems(std::function<void(const std::unordered_map<Item, Item::Info, Item::Hasher>&)> resolvedItemsHandler);

        void addListener(std::weak_ptr<Listener> listener);

        void removeListener(std::weak_ptr<Listener> listener);

        AvahiBrowseClient(const AvahiBrowseClient&) = delete;
        AvahiBrowseClient& operator=(const AvahiBrowseClient&) = delete;

        /*
         * For test purposes only
         */

        friend struct TestEventTrigger;

        struct TestEventTrigger {
            AvahiBrowseClient* client = nullptr;

            void onBrowserFailure(const Error& error) const;
            void onBrowserNew(const Item& item) const;
            void onBrowserRemove(const Item& item) const;
            void onResolveFailure(const Item& item, const Error& error) const;
            void onNewResolved(const Item& item, const Item::Info& info) const;
        };

        static void initForTest();

        TestEventTrigger getTestEventTrigger();

    public: // IAvahiBrowseClient
        IResolvedItemsChangedSignal& resolvedItemsChangedSignal() override;

    private:
        enum class InitSate {
            NOT_INITED,
            INITED,
            INITED_FOR_TEST
        };

        std::list<std::weak_ptr<Listener>> copyListeners();

        void onBrowserFailure(const Error& error);
        void onBrowserNew(const Item& item);
        void onBrowserRemove(const Item& item);
        void onResolveFailure(const Item& item, const Error& error);
        void onResolved(const Item& item, const Item::Info& info);

        void startAvahiPollLoop();
        void startBrowsing();
        void stopBrowsing();
        bool isBrowsing();
        void clean();

        /*
         * Helpers are used to set AvahiBrowseClient (resolveCallback and browseCallback) methods as callbacks for avahi.
         * "this" is passed as userdata
         */

        static void resolveCallbackHelper(
            AvahiSServiceResolver* resolver,
            AVAHI_GCC_UNUSED AvahiIfIndex interface,
            AVAHI_GCC_UNUSED AvahiProtocol protocol,
            AvahiResolverEvent event,
            const char* name,
            const char* type,
            const char* domain,
            const char* host_name,
            const AvahiAddress* address,
            uint16_t port,
            AvahiStringList* txt,
            AvahiLookupResultFlags flags,
            AVAHI_GCC_UNUSED void* userdata);

        static void browseCallbackHelper(
            AvahiSServiceBrowser* browser,
            AvahiIfIndex ifIndex,
            AvahiProtocol protocol,
            AvahiBrowserEvent event,
            const char* name,
            const char* type,
            const char* domain,
            AVAHI_GCC_UNUSED AvahiLookupResultFlags flags,
            void* userdata);

        /*
         * Avahi events handlers
         */

        void resolveCallback(
            AvahiResolverEvent event,
            const char* name,
            const char* type,
            const char* domain,
            const char* host_name,
            const AvahiAddress* address,
            uint16_t port,
            AvahiStringList* txt);

        void browseCallback(
            AvahiIfIndex ifIndex,
            AvahiProtocol protocol,
            AvahiBrowserEvent event,
            const char* name,
            const char* type,
            const char* domain);

        static InitSate initState_;

        const bool ignoreIpv6_{false};
        AvahiSimplePoll* simple_poll_ = nullptr;
        AvahiServer* server_ = nullptr;
        AvahiSServiceBrowser* sb_ = nullptr;

        std::string serviceType_ = "_yandexio._tcp."; // TODO configurable
        const std::optional<std::string> onlyNameToResolve_;
        /*
         * Can be accessed only in callbacks thread
         */
        std::unordered_map<Item, Item::Info, Item::Hasher> resolved_items_;

        std::mutex listenersMutex_;
        std::list<std::weak_ptr<Listener>> listeners_;

        mutable std::mutex mutex_;

        std::thread avahiServerThread_;
        bool isBrowsing_ = false;

        std::shared_ptr<quasar::ICallbackQueue> callbackQueue_;
        quasar::SignalExternal<IResolvedItemsChangedSignal> resolvedItemsChangedSignal_;
    };

    AvahiBrowseClient::Item::Info makeInfo(const char* hostname, int port, const char* address, AvahiStringList* txtList);

} // namespace glagol
