#include "backend_api.h"
#include <yandex_io/libs/http_client/http_client.h>

using namespace glagol;
using namespace quasar;

namespace {
    GroupRole roleFromString(const std::string& src) {
        if (src == "leader") {
            return RoleNest::LEADER;
        }
        if (src == "follower") {
            return RoleNest::FOLLOWER;
        }
        return RoleNest::STAND_ALONE;
    }

    std::shared_ptr<ISimpleHttpClient> makeDefaultHttpClient(std::shared_ptr<YandexIO::IDevice> device) {
        auto result = std::make_shared<HttpClient>("gsdk-backend", std::move(device));
        result->setTimeout(std::chrono::seconds{5}); // 5s timeout for backend access -- we are ready to wait here
        return result;
    }
} // namespace

BackendApi::BackendApi(std::shared_ptr<quasar::IAuthProvider> authProvider, std::shared_ptr<ISimpleHttpClient> backendClient, Settings settings)
    : backendClient_(std::move(backendClient))
    , authProvider_(std::move(authProvider))
{
    setSettings(std::move(settings));
}

BackendApi::BackendApi(std::shared_ptr<quasar::IAuthProvider> authProvider, std::shared_ptr<YandexIO::IDevice> device)
    : BackendApi(std::move(authProvider), makeDefaultHttpClient(std::move(device)), Settings{})
{
}

BackendApi::BackendApi(std::shared_ptr<quasar::IAuthProvider> authProvider, Settings settings, std::shared_ptr<YandexIO::IDevice> device)
    : BackendApi(std::move(authProvider), makeDefaultHttpClient(std::move(device)), std::move(settings))
{
}

bool BackendApi::setSettings(Settings settings) {
    if (settings_ == settings) {
        return false;
    }
    settings_ = std::move(settings);
    return true;
}

BackendApi::Settings BackendApi::settings() const {
    return settings_;
}

BackendApi::DevicesMap BackendApi::getConnectedDevicesList() {
    HttpClient::HttpResponse response(0, "", "");
    try {
        response = backendClient_->get("device-list", settings_.url + "/glagol/device_list", getHeaders());
    } catch (const std::runtime_error& error) {
        throw Exception(error.what());
    }
    auto jsonBody = checkAndParseResponse(response);
    std::map<DeviceId, Device> result;
    for (const auto& deviceJson : jsonBody["devices"]) {
        result.emplace(IBackendApi::fromJson(deviceJson));
    }
    return result;
}

std::string BackendApi::getToken(const DeviceId& deviceId) {
    HttpClient::HttpResponse response(0, "", "");
    try {
        response = backendClient_->get("token", settings_.url + "/glagol/token?device_id=" + deviceId.id + "&platform=" + deviceId.platform, getHeaders());
    } catch (const std::runtime_error& error) {
        throw Exception(error.what());
    }
    auto jsonBody = checkAndParseResponse(response);
    return jsonBody["token"].asString();
}

bool BackendApi::checkToken(const std::string& token) {
    HttpClient::HttpResponse response(0, std::string());
    try {
        response = backendClient_->post("check-token", settings_.url + "/glagol/check_token", token, getHeaders());
    } catch (const std::runtime_error& error) {
        throw Exception(error.what());
    }
    auto jsonBody = checkAndParseResponse(response);
    return jsonBody["valid"].asBool();
}

IBackendApi::TokenCheckResult BackendApi::checkToken2(const std::string& token) {
    HttpClient::HttpResponse response(0, std::string());
    try {
        response = backendClient_->post("check-token", settings_.url + "/glagol/v2.0/check_token", token, getHeaders());
    } catch (const std::runtime_error& error) {
        throw Exception(error.what());
    }
    auto jsonBody = checkAndParseResponse(response);
    return {.owner = jsonBody["owner"].asBool(), .guest = jsonBody["guest"].asBool()};
}

Json::Value BackendApi::checkAndParseResponse(const ISimpleHttpClient::HttpResponse& response) {
    YIO_LOG_DEBUG("GSDK. Response" << response.body);
    if (response.responseCode >= 200 && response.responseCode <= 299) {
        try {
            auto registerBody = quasar::parseJson(response.body);
            if ("ok" != registerBody["status"].asString()) {
                throw Exception("Non-ok response status");
            }
            return registerBody;
        } catch (const Json::Exception& exception) {
            throw Exception("Json parsing error: " + std::string(exception.what()));
        }
    } else if (response.responseCode == 403 && quasar::getStringSafe(response.body, ".message") == "AUTH_TOKEN_INVALID") {
        authProvider_->requestAuthTokenUpdate("BackendApi checkAndParseResponse");
    }
    throw Non200ResponseCodeException(response.responseCode);
}

quasar::HttpClient::Headers BackendApi::getHeaders() const {
    return quasar::HttpClient::Headers{
        {"Authorization", "OAuth " + settings_.token},
        {"Content-Type", "application/json"},
    };
}

using NetworkInfo = IBackendApi::NetworkInfo;

IBackendApi::Device::Device(const Json::Value& json)
    : name(json["name"].asString())
    , activationCode(json["activation_code"].asString())
    , promocodeActivated(json["promocode_activated"].asBool())
    , guestMode(tryGetBool(json, "guestMode", false))
    , config(json["config"])
    , glagol(json["glagol"])
    , group(tryGetJson(json, "group", Json::Value()))
    , networkInfo(NetworkInfo::fromJson(tryGetJson(json, "networkInfo", Json::Value())))
{
    for (const auto& tag : json["tags"]) {
        tags.push_back(tag.asString());
    }
}

GroupRole IBackendApi::Device::roleFromGroupJson(const Json::Value& config, const std::string& id, const std::string& platform) {
    try {
        if (config.isMember("devices")) {
            const auto& devices = config["devices"];
            if (devices.isArray()) {
                // id, platform, role
                for (unsigned i = 0; i < devices.size(); ++i) {
                    const Json::Value& device = devices[i];
                    if (id == device["id"].asString() && platform == device["platform"].asString()) {
                        return roleFromString(device["role"].asString());
                    }
                }
            }
        }
    } catch (const Json::Exception& e) {
    }
    return RoleNest::STAND_ALONE;
}

std::pair<DeviceId, IBackendApi::Device> IBackendApi::fromJson(const Json::Value& json) {
    Device device(json);
    DeviceId id{.id = json["id"].asString(), .platform = json["platform"].asString()};
    device.tandemRole = Device::roleFromGroupJson(device.group, id.id, id.platform);
    return {std::move(id), std::move(device)};
}

Json::Value IBackendApi::NetworkInfo::serialize() const {
    Json::Value result;
    result["ts"] = ts;
    result["ip_addresses"] = quasar::vectorToJson(IPs);
    result["mac_addresses"] = quasar::vectorToJson(MACs);
    result["external_port"] = externalPort;
    if (!wifiSsid.empty()) {
        result["wifi_ssid"] = wifiSsid;
    }
    if (stereopairRole != RoleNest::STAND_ALONE) {
        result["stereopair"] = stereopairRole == RoleNest::LEADER ? "leader" : "follower";
    }
    return result;
}

NetworkInfo IBackendApi::NetworkInfo::fromJson(const Json::Value& src) {
    try {
        NetworkInfo result;
        result.ts = src["ts"].asInt();
        result.IPs = jsonToVec<std::string>(src["ip_addresses"]);
        result.MACs = jsonToVec<std::string>(src["mac_addresses"]);
        result.externalPort = src["external_port"].asInt();
        if (src.isMember("wifi_ssid")) {
            result.wifiSsid = src["wifi_ssid"].asString();
        }
        if (src.isMember("stereopair")) {
            result.stereopairRole = roleFromString(src["stereopair"].asString());
        }
        return result;
    } catch (...) {
        YIO_LOG_WARN("Broken NetworkInfo wasnt deserialized " << src);
    }
    return NetworkInfo();
}

Json::Value IBackendApi::Device::serialize(const DeviceId& deviceId) const {
    Json::Value result;
    result["id"] = deviceId.id;
    result["platform"] = deviceId.platform;
    result["activation_code"] = activationCode;
    result["promocode_activated"] = promocodeActivated;
    result["guestMode"] = guestMode;
    result["config"] = config.serialize();
    result["glagol"] = glagol.serialize();
    result["tags"] = Json::Value{Json::arrayValue};
    result["group"] = group;
    result["networkInfo"] = networkInfo.serialize();
    return result;
}

bool IBackendApi::Device::operator==(const Device& other) const {
    return name == other.name &&
           activationCode == other.activationCode &&
           promocodeActivated == other.promocodeActivated &&
           config == other.config &&
           glagol == other.glagol &&
           tags == other.tags;
}

bool IBackendApi::Device::operator!=(const Device& other) const {
    return !(*this == other);
}

IBackendApi::Device::Config::Config(const Json::Value& json)
    : masterDevice(json["masterDevice"])
    , name(json["name"].asString())
{
}

Json::Value IBackendApi::Device::Config::serialize() const {
    Json::Value result;
    result["masterDevice"] = masterDevice.serialize();
    result["name"] = name;
    return result;
}

bool BackendApi::Device::Config::operator==(const Config& other) const {
    return masterDevice == other.masterDevice && name == other.name;
}

BackendApi::Device::Config::MasterDevice::MasterDevice(const Json::Value& json)
    : id(json["deviceId"].asString())
{
}

Json::Value BackendApi::Device::Config::MasterDevice::serialize() const {
    Json::Value result;
    result["deviceId"] = id;
    return result;
}

bool BackendApi::Device::Config::MasterDevice::operator==(const MasterDevice& other) const {
    return id == other.id;
}

IBackendApi::Device::Glagol::Glagol(const Json::Value& json)
    : security(json["security"])
{
}

Json::Value IBackendApi::Device::Glagol::serialize() const {
    Json::Value result;
    result["security"] = security.serialize();
    return result;
}

bool IBackendApi::Device::Glagol::operator==(const Glagol& other) const {
    return security == other.security;
}

IBackendApi::Device::Glagol::Security::Security(const Json::Value& json)
    : serverCertificate(json["server_certificate"].asString())
    , serverPrivateKey(json["server_private_key"].asString())
{
}

Json::Value IBackendApi::Device::Glagol::Security::serialize() const {
    Json::Value result;
    result["server_certificate"] = serverCertificate;
    result["server_private_key"] = serverPrivateKey;
    return result;
}

bool IBackendApi::Device::Glagol::Security::operator==(const Security& other) const {
    return serverCertificate == other.serverCertificate && serverPrivateKey == other.serverPrivateKey;
}

bool IBackendApi::Device::Glagol::Security::filledForServer() const {
    return !(serverCertificate.empty() || serverPrivateKey.empty());
}

/*********************************************************/

bool BackendSettings::operator==(const BackendSettings& other) const {
    return url == other.url && token == other.token;
}

BackendSettings::operator bool() const {
    return !operator!();
}

bool BackendSettings::operator!() const {
    return url.empty() || token.empty();
}
