#include "cached_backend_api.h"

#include "backend_api.h"

namespace glagol {

    constexpr std::chrono::seconds TOKEN_CACHE_TIME{180};
    constexpr std::chrono::seconds MAX_DEVICES_LIST_WAIT_TIME{45}; // AccountDevices::scheduleUpdate for 30 seconds, 15 seconds more for http request

    class CachedBackendApi: public glagol::BackendApi2 {
        struct Token {
            std::string token;
            std::chrono::steady_clock::time_point requestTime;

            Token(const std::string& value)
                : token(value)
                , requestTime(std::chrono::steady_clock::now())
                      {};

            bool isOutdated() const {
                return std::chrono::steady_clock::now() - requestTime > TOKEN_CACHE_TIME;
            };
        };

        std::shared_ptr<YandexIO::ITelemetry> telemetry_;
        mutable std::mutex devicesListMutex;
        std::condition_variable deviceListCond;
        DevicesMap devicesList;
        std::vector<DevicesMap::const_iterator> devicesIds; // sorted vector to search over ids only. DevicesMap::key is {.id, .platform}
        std::atomic_bool needNewDeviceList = false;

        std::mutex tokensMutex;
        using DeviceId = glagol::DeviceId;
        std::unordered_map<DeviceId, Token, DeviceId::Hasher> tokens;

        glagol::BackendApi realBackendApi;
        std::function<void()> deviceListRequest;

        static std::string toJson(const DeviceId& id) {
            Json::Value json(Json::objectValue);
            json["id"] = id.id;
            json["platform"] = id.platform;
            return quasar::jsonToString(json);
        }

        std::string getTokenFromBackend(const DeviceId& deviceId) {
            std::string token = realBackendApi.getToken(deviceId);
            telemetry_->reportEvent("gsdkDiscoveryTokenGetSuccess", toJson(deviceId));
            std::lock_guard<std::mutex> lock(tokensMutex);
            tokens.emplace(deviceId, token);
            return token;
        }

    public:
        CachedBackendApi(std::shared_ptr<YandexIO::IDevice> device,
                         const glagol::BackendApi::Settings& settings,
                         std::shared_ptr<quasar::IAuthProvider> authProvider,
                         std::function<void()> deviceListRequestCb,
                         const DevicesMap& devices)
            : telemetry_(device->telemetry())
            , devicesList(devices)
            , realBackendApi(std::move(authProvider), settings, std::move(device))
            , deviceListRequest(std::move(deviceListRequestCb))
        {
        }

        ~CachedBackendApi() {
            needNewDeviceList = false;
            deviceListCond.notify_all();
        }

        void setDevicesList(const DevicesMap& newDevicesList) override {
            std::lock_guard<std::mutex> lock(devicesListMutex);
            devicesList = newDevicesList;
            // prepare 'set' to search only for id
            devicesIds.clear();
            devicesIds.reserve(devicesList.size());
            for (auto it = std::cbegin(devicesList), end = std::cend(devicesList); it != end; ++it) {
                devicesIds.emplace_back(it);
            }
            std::sort(std::begin(devicesIds), std::end(devicesIds),
                      [](auto a, auto b) {
                          return a->first.id < b->first.id;
                      });
            needNewDeviceList = false;
            deviceListCond.notify_all();
        }

        DevicesMap getConnectedDevicesList() override {
            std::unique_lock<std::mutex> lock(devicesListMutex);
            if (needNewDeviceList) { //'if' for optional telemetry on waiting
                YIO_LOG_DEBUG("Wait for new devices list");
                auto latencyPoint_ = telemetry_->createLatencyPoint();
                deviceListCond.wait_for(lock, MAX_DEVICES_LIST_WAIT_TIME,
                                        [this]() {
                                            return !needNewDeviceList;
                                        });
                telemetry_->reportLatency(latencyPoint_, "GlagolWaitConnectedDevicesList");
                YIO_LOG_DEBUG("New devices list came");
            }
            return devicesList;
        }

        std::optional<DeviceId> hasDeviceInDeviceList(const std::string& id) const override {
            std::lock_guard<std::mutex> lock(devicesListMutex);
            auto it = std::lower_bound(
                std::begin(devicesIds), std::end(devicesIds), id,
                [](auto it, const auto& str) {
                    return it->first.id < str;
                });
            if (it != std::end(devicesIds) && (*it)->first.id == id) {
                return (*it)->first;
            };
            return std::nullopt;
        }

        void setSettings(const glagol::BackendApi::Settings& settings) override {
            realBackendApi.setSettings(settings);
        }

        std::string getToken(const DeviceId& deviceId) override {
            {
                std::lock_guard<std::mutex> lock(tokensMutex);
                auto tokenIter = tokens.find(deviceId);
                if (tokenIter == tokens.end()) { // graceful way to handle lock
                } else if (tokenIter->second.isOutdated()) {
                    tokens.erase(tokenIter);
                } else {
                    return tokenIter->second.token;
                }
            }
            return getTokenFromBackend(deviceId);
        }

        bool checkToken(const std::string& /*token*/) override {
            throw std::runtime_error("CachedBackendApi: method checkToken is not supported");
            return false;
        }

        IBackendApi::TokenCheckResult checkToken2(const std::string& /*token*/) override {
            throw std::runtime_error("CachedBackendapi: method checkToken2 is not supported");
            return {};
        }

        void invalidToken(const DeviceId& id) override {
            std::lock_guard<std::mutex> lock(tokensMutex);
            tokens.erase(id);
        }

        void invalidCert(const DeviceId& id) override {
            needNewDeviceList = true;
            YIO_LOG_INFO("Probably updated cert for device '" << id.id << "'");
            deviceListRequest();
        }
    };

    std::shared_ptr<BackendApi2> createCachedBackendApi(std::shared_ptr<YandexIO::IDevice> device,
                                                        const glagol::BackendSettings& settings,
                                                        std::shared_ptr<quasar::IAuthProvider> authProvider,
                                                        std::function<void()> deviceListRequestCb,
                                                        const IBackendApi::DevicesMap& devices) {
        return std::make_shared<CachedBackendApi>(device, settings, authProvider, deviceListRequestCb, devices);
    }

} // namespace glagol
