#pragma once
#include "conversation_model.h"
#include "i_backend_api.h"
#include "i_discovery.h"
#include "quasar_includes.h"

#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/threading/lifetime.h>

#include <map>
#include <string>

namespace glagol {

    class Connector {
    public:
        enum class State {
            STOPPED,
            DISCOVERING,
            REQUESTING_BACKEND, // device is discovered via mDNS. JWT and public_key is being requested
            CONNECTING,
            CONNECTED,
            DISCONNECTING, // waiting for WebsocketClient to close connection

            LAST_ENUM // is used for static asserts. Not used in runtime
        };
        class StateWrapper {
        public:
            State getState() const;
            void set(State state);
            std::function<void(State stateValue)> onChangedCallback;
            bool operator==(State state) const;
            bool operator!=(State state) const;

        private:
            std::atomic<State> state_{State::STOPPED};
        };

        struct Settings {
            struct Backend {
                quasar::RetryDelayCounter::Settings retryDelay;
            } backend;
            struct Websocket {
                std::chrono::milliseconds retryDelay = std::chrono::seconds(5);
                std::chrono::milliseconds pingTimeout = std::chrono::seconds(5);
            } websocket;
            DeviceId deviceId;
        };

        Connector(std::shared_ptr<IDiscovery> discovery,
                  std::shared_ptr<IBackendApi> backendApi,
                  std::shared_ptr<YandexIO::ITelemetry> telemetry,
                  std::string origin = std::string());
        ~Connector();

        void connect(Settings settings);

        void disconnectAsync();

        void setOnStateChangedCallback(std::function<void(State)> onStateChanged);

        void setOnMessageCallback(std::function<void(const model::IncomingMessage&)> onMessage);

        void setOnPongCallback(std::function<void()> onPong);

        std::string send(const Json::Value& payload);

        std::string send(const Json::Value& payload, std::function<void(const model::IncomingMessage&)> onResponse);

        model::IncomingMessage sendSync(const Json::Value& payload,
                                        std::chrono::milliseconds timeOut = std::chrono::seconds(1));

        /*
         * Must be called inside main thread (in any callback)
         */
        const Settings& getSettings() const;

        /*
         * Must be called inside main thread (in any callback)
         */
        State getState() const;

        // Testing purposes only
        bool waitForState(State state, std::chrono::milliseconds timeOut = std::chrono::seconds(5)) const;

    private:
        const std::string metricaOrigin_;
        Json::Value metricaContextJson_;
        std::string metricaContext_;

        Json::Value getConnectMetricaContext() const;
        void updateMetricaContext();

        void onDiscoveryResultChanged(const IDiscovery::Result& result);

        void onWsConnected();

        void onWsDisconnected(quasar::Websocket::ConnectionInfo connectionInfo);

        void onWsPong();

        void getCertAndToken();

        void handleMessage(const std::string& message);

        Json::Value makeMessage(const Json::Value& value);

    private:
        quasar::Lifetime lifetime_;
        std::shared_ptr<YandexIO::ITelemetry> telemetry_;

        Settings settings_;
        std::shared_ptr<IDiscovery> discovery_;
        quasar::SignalConnectionId discoverySignalId_;
        StateWrapper state_;

        quasar::WebsocketClient wsClient_;
        quasar::WebsocketClient::Settings wsSettings_;
        std::string jwtToken_;
        quasar::RetryDelayCounter retryDelayCounter_;

        std::function<void(const model::IncomingMessage&)> onMessage_;
        std::map<std::string, std::function<void(const model::IncomingMessage&)>> pendingResponses_;

        mutable quasar::SteadyConditionVariable stateWakeUpVar_;

        std::atomic<bool> stopped_{false};
        std::shared_ptr<quasar::ICallbackQueue> callbackQueue_;
        std::function<void()> onPong_;
        std::shared_ptr<IBackendApi> backendApi_;
    };

} // namespace glagol

namespace std {
    string to_string(glagol::Connector::State /*state*/);
} // namespace std
