#pragma once
#include "conversation_model.h"
#include "discovered_item.h"
#include "i_backend_api.h"
#include "quasar_includes.h"

#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/threading/lifetime.h>

#include <map>
#include <string>

namespace glagol {
    namespace ext {

        class Connector {
        public:
            enum class State {
                STOPPED,
                DISCOVERING,
                REQUESTING_BACKEND, // device is discovered via mDNS. JWT and public_key is being requested
                CONNECTING,
                CONNECTED,
                DISCONNECTING, // waiting for WebsocketClient to close connection

                LAST_ENUM // is used for static asserts. Not used in runtime
            };

            struct Settings {
                struct Backend {
                    quasar::RetryDelayCounter::Settings retryDelay;
                } backend;
                struct Websocket {
                    std::chrono::milliseconds retryDelay = std::chrono::seconds(5);
                    std::chrono::milliseconds pingTimeout = std::chrono::seconds(5);
                } websocket;
            };

            using OnStateChangedCallback = std::function<void(State)>;

            Connector(std::shared_ptr<IBackendApi> backendApi,
                      std::shared_ptr<YandexIO::ITelemetry> telemetry,
                      std::string origin = std::string());
            ~Connector();

            void connect(DiscoveredItemPtr /*device*/, Settings /*settings*/);

            void disconnectAsync();

            void setOnStateChangedCallback(OnStateChangedCallback /*onStateChanged*/, State knownState = State::STOPPED);

            void setOnMessageCallback(std::function<void(const model::IncomingMessage&)> onMessage);

            void setOnPongCallback(std::function<void()> onPong);

            std::string send(const Json::Value& payload);

            std::string send(const Json::Value& payload, std::function<void(const model::IncomingMessage&)> onResponse);

            model::IncomingMessage sendSync(const Json::Value& payload,
                                            std::chrono::milliseconds timeOut = std::chrono::seconds(1));

            /*
             * Must be called inside main thread (in any callback)
             */
            const Settings& getSettings() const;

            /*
             * Must be called inside main thread (in any callback)
             */
            State getState() const;

            // Testing purposes only
            bool waitForState(State state, std::chrono::milliseconds timeOut = std::chrono::seconds(5)) const;

        private:
            class StateWrapper {
            public:
                State getState() const;
                void set(State state);
                void setOnChange(OnStateChangedCallback /*cb*/, State /*knownState*/);
                bool waitFor(State state, std::chrono::milliseconds timeOut) const;
                bool operator==(State state) const;
                bool operator!=(State state) const;

            private:
                mutable std::mutex mutex_;
                mutable quasar::SteadyConditionVariable wakeUpVar_;
                OnStateChangedCallback onChanged_;
                std::atomic<State> state_{State::STOPPED};
            };

            const std::string metricaOrigin_;
            Json::Value metricaContextJson_;
            std::string metricaContext_;

            Json::Value getConnectMetricaContext() const;
            void updateMetricaContext();

            void onDiscovery(DiscoveredItem::ConnectionData /*result*/);

            void onWsConnected();

            void onWsDisconnected(quasar::Websocket::ConnectionInfo connectionInfo);

            void onWsPong();

            void getToken();

            void handleMessage(const std::string& message);

            Json::Value makeMessage(const Json::Value& value);

        private:
            quasar::Lifetime lifetime_;
            std::shared_ptr<YandexIO::ITelemetry> telemetry_;

            Settings settings_;
            DiscoveredItemPtr destination_;
            StateWrapper state_;

            quasar::WebsocketClient wsClient_;
            quasar::WebsocketClient::Settings wsSettings_;
            std::string jwtToken_;
            quasar::RetryDelayCounter retryDelayCounter_;

            std::function<void(const model::IncomingMessage&)> onMessage_;
            std::map<std::string, std::function<void(const model::IncomingMessage&)>> pendingResponses_;

            std::atomic<bool> stopped_{false};
            std::shared_ptr<quasar::ICallbackQueue> callbackQueue_;
            std::function<void()> onPong_;
            std::shared_ptr<IBackendApi> backendApi_;
        };

    } // namespace ext
} // namespace glagol

namespace std {
    string to_string(glagol::ext::Connector::State /*state*/);
} // namespace std
