#include "discovery.h"

#include "quasar_includes.h"
#include "avahi_wrapper/avahi_browse_client.h"

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/json_utils/json_utils.h>

using namespace quasar;
using namespace glagol;

namespace {
    std::string protoToText(Discovery::Result::Protocol proto)
    {
        std::string result;
        switch (proto) {
            case Discovery::Result::Protocol::ipv4:
                result = "ipv4";
                break;
            case Discovery::Result::Protocol::ipv6:
                result = "ipv6";
                break;
        }
        return result;
    }

    std::string printResultItem(const DeviceId& deviceId, const Discovery::Result::Item& item, bool discovered)
    {
        Json::Value json;
        json["name"] = item.name;
        json["device_id"] = deviceId.id + ":" + deviceId.platform;
        json["host"] = item.host;
        json["address"] = item.address;
        json["port"] = item.port;
        json["uri"] = item.uri;
        json["account"] = item.isAccountDevice;
        json["cluster"] = (item.cluster ? "yes" : "no");
        json["proto"] = protoToText(item.protocol);
        return jsonToString(json) + (discovered ? " (discovered)" : "");
    }

    std::string printMyDevice(const DeviceId& deviceId, const BackendApi::Device& device)
    {
        Json::Value json;
        json["name"] = device.name;
        json["device_id"] = deviceId.id + ":" + deviceId.platform;
        return jsonToString(json);
    }

} // namespace

Discovery::Discovery(
    std::shared_ptr<IAccountDevices> accountDevices,
    std::shared_ptr<IAvahiBrowseClient> browseClient,
    Settings settings,
    std::function<void(const Result&)> onResultChanged)
    : Discovery(
          std::make_shared<quasar::NamedCallbackQueue>("Discovery"),
          std::move(accountDevices),
          std::move(browseClient),
          std::move(settings))
{
    devicesAroundHasChangedSignal_.connect(
        [onResultChanged{std::move(onResultChanged)}](const Result& r) {
            onResultChanged(r);
        }, lifetime_);
}

Discovery::Discovery(
    std::shared_ptr<quasar::ICallbackQueue> lifecycle,
    std::shared_ptr<IAccountDevices> accountDevices,
    std::shared_ptr<IAvahiBrowseClient> browseClient,
    Settings settings)
    : lifecycle_(std::move(lifecycle))
    , accountDevices_(std::move(accountDevices))
    , browseClient_(std::move(browseClient))
    , settings_(std::move(settings))
    , devicesAroundHasChangedSignal_(
          [this](bool /*onConnect*/) {
              std::lock_guard<std::mutex> lock(mutex_);
              return publicResult_;
          }, lifetime_)
{
    YIO_LOG_INFO("GSDK. Discovery init");
    if (accountDevices_) {
        accountDevices_->deviceListChangedSignal().connect(
            [this](const BackendApi::DevicesMap& deviceMap)
            {
                myDevices_ = deviceMap;
                notifyAboutDiscoveredDevices();
            }, lifetime_, lifecycle_);
    }

    if (browseClient_) {
        browseClient_->resolvedItemsChangedSignal().connect(
            [this](const IAvahiBrowseClient::ResolvedItems& resolvedItems) {
                onUpdateOriginResult(resolvedItems);
            }, lifetime_, lifecycle_);
    }
}

Discovery::~Discovery()
{
    lifetime_.die();
}

Discovery::Result Discovery::devicesAround() const noexcept {
    std::lock_guard<std::mutex> lock(mutex_);
    return publicResult_;
}

Discovery::IDevicesAroundHasChangedSignal& Discovery::devicesAroundHasChangedSignal()
{
    return devicesAroundHasChangedSignal_;
}

void Discovery::onUpdateOriginResult(const IAvahiBrowseClient::ResolvedItems& resolvedItems)
{
    Y_ENSURE_THREAD(lifecycle_);

    bool fChanged = false;
    std::string uri;
    uri.reserve(128);

    std::set<const Result::Item*> knownItems;
    for (const auto& [item, info] : resolvedItems) {
        if (item.domain != "local") {
            YIO_LOG_DEBUG("GSDK. Wrong Domain: " + item.domain);
            continue;
        }
        if (item.type != settings_.serviceType) {
            YIO_LOG_DEBUG("GSDK. Wrong Service Type: " + item.type);
            continue;
        }
        if (item.name.substr(0, settings_.serviceNamePrefix.size()) != settings_.serviceNamePrefix) {
            YIO_LOG_DEBUG("GSDK. Wrong Service Name Prefix: " + item.type);
            continue;
        }
        if (!info.txt.count("deviceId") || !info.txt.count("platform")) {
            YIO_LOG_DEBUG("GSDK. No deviceId or platform in txt");
            continue;
        }
        if (!settings_.filter.platforms.empty() && !settings_.filter.platforms.count(info.txt.at("platform"))) {
            YIO_LOG_DEBUG("GSDK. Skipping due to platforms filter");
            continue;
        }
        if (!settings_.filter.ids.empty() && !settings_.filter.ids.count(info.txt.at("deviceId"))) {
            YIO_LOG_DEBUG("GSDK. Skipping due to ids filter");
            continue;
        }

        DeviceId deviceId{info.txt.at("deviceId"), info.txt.at("platform")};

        // ipv6 addresses have ':' in them, like '2a02:6bf:fff0:0:ce4b:73ff:fe23:10c6'
        // these need to be wrapped in brackets, see https://st.yandex-team.ru/ALICE-2420
        Result::Protocol protocol = glagol::Discovery::Result::Protocol::ipv4;
        if (info.address.find(':') != std::string::npos) {
            YIO_LOG_DEBUG("GSDK wrapping ipv6 address: " + info.address);
            protocol = glagol::Discovery::Result::Protocol::ipv6;
            uri = "[";
            uri += info.address;
            uri += "]";
        } else {
            uri = info.address;
        }
        uri += ":";
        uri += std::to_string(info.port);

        auto it = originResult_.items.find(deviceId);
        if (it != originResult_.items.end() &&
            it->second.name == item.name &&
            it->second.host == info.hostname &&
            it->second.uri == uri)
        {
            // no changes
            knownItems.insert(&it->second);
            continue;
        }

        fChanged = true;
        auto res = originResult_.items.insert_or_assign(
            deviceId,
            Result::Item{
                .name = item.name,
                .host = info.hostname,
                .address = info.address,
                .port = info.port,
                .uri = uri,
                .isAccountDevice = false,
                .cluster = (info.txt.count("cluster") && info.txt.at("cluster") == "yes"),
                .protocol = protocol,
            });
        knownItems.insert(&res.first->second);
    }
    if (knownItems.size() != originResult_.items.size()) {
        for (auto it = originResult_.items.begin(); it != originResult_.items.end();) {
            if (knownItems.count(&it->second)) {
                ++it;
            } else {
                fChanged = true;
                it = originResult_.items.erase(it);
            }
        }
    }

    if (fChanged) {
        notifyAboutDiscoveredDevices();
    }
}

void Discovery::notifyAboutDiscoveredDevices()
{
    Y_ENSURE_THREAD(lifecycle_);

    Result newResult = originResult_;
    if (accountDevices_) {
        for (auto it = newResult.items.begin(); it != newResult.items.end();) {
            auto jt = myDevices_.find(it->first);
            if (jt != myDevices_.end()) {
                it->second.isAccountDevice = true;
                if (it->second.name != jt->second.name) {
                    it->second.name = jt->second.name;
                }
                ++it;
            } else {
                if (settings_.accountOnly) {
                    it = newResult.items.erase(it);
                } else {
                    ++it;
                }
            }
        }
    }

    std::unique_lock<std::mutex> lock(mutex_);
    bool fChanged = (newResult.items != publicResult_.items);
    if (fChanged) {
        if (YIO_LOG_DEBUG_ENABLED()) {
            YIO_LOG_DEBUG("Notify about discovered devices:");
            for (const auto& [deviceId, item] : newResult.items) {
                YIO_LOG_DEBUG("    (" << (item.isAccountDevice ? "+" : " ") << ") " << printResultItem(deviceId, item, !publicResult_.items.count(deviceId)));
            }
            for (const auto& [deviceId, device] : myDevices_) { // FIXME: ordered maps intersection
                if (newResult.items.count(deviceId) == 0) {
                    YIO_LOG_DEBUG("        " << printMyDevice(deviceId, device) << " (undiscovered)");
                }
            }
        }
        publicResult_ = std::move(newResult);
    }
    lock.unlock();

    if (fChanged) {
        devicesAroundHasChangedSignal_.emit();
    }
}
