#pragma once

#include "device_id.h"
#include "role_types.h"

#include <json/json.h>

#include <map>
#include <stdexcept>
#include <string>
#include <vector>

namespace glagol {

    class IBackendApi {
    public:
        struct NetworkInfo {
            std::uint64_t ts{0};
            std::vector<std::string> IPs;
            std::vector<std::string> MACs;
            std::string wifiSsid;
            int externalPort{0};
            GroupRole stereopairRole{RoleNest::STAND_ALONE};

            Json::Value serialize() const;
            static NetworkInfo fromJson(const Json::Value& src);
        };

        struct Device {
            struct Config {
                Config() = default;
                explicit Config(const Json::Value& json);
                Json::Value serialize() const;
                bool operator==(const Config& other) const;

                struct MasterDevice {
                    MasterDevice() = default;
                    explicit MasterDevice(const Json::Value& json);
                    Json::Value serialize() const;
                    bool operator==(const MasterDevice& other) const;

                    std::string id;
                };
                MasterDevice masterDevice;
                std::string name;
            };

            struct Glagol {
                Glagol() = default;
                explicit Glagol(const Json::Value& json);
                Json::Value serialize() const;
                bool operator==(const Glagol& other) const;

                struct Security {
                    Security() = default;
                    explicit Security(const Json::Value& json);
                    Json::Value serialize() const;
                    bool operator==(const Security& other) const;
                    bool filledForServer() const;

                    std::string serverCertificate;
                    std::string serverPrivateKey;
                };
                Security security;
            };

            std::string name;
            std::string activationCode;
            bool promocodeActivated;
            bool guestMode;

            Config config;
            Glagol glagol;
            Json::Value group;
            NetworkInfo networkInfo;
            GroupRole tandemRole{RoleNest::STAND_ALONE};
            std::vector<std::string> tags;

            Device() = default;
            explicit Device(const Json::Value& json);

            Json::Value serialize(const DeviceId& /*deviceId*/) const;
            bool operator==(const Device& other) const;
            bool operator!=(const Device& other) const;

            static GroupRole roleFromGroupJson(const Json::Value& /*config*/, const std::string& id, const std::string& platform);
        };

        static std::pair<DeviceId, Device> fromJson(const Json::Value& /*json*/);
        using DevicesMap = std::map<DeviceId, Device>;

        class Exception: public std::runtime_error {
        public:
            explicit Exception(const std::string& message)
                : std::runtime_error(message) {
            }
            virtual ~Exception() = default;
        };

        class Non200ResponseCodeException: public Exception {
        public:
            explicit Non200ResponseCodeException(int code)
                : Exception("Non-200 response code")
                , code_(code)
            {
            }
            virtual ~Non200ResponseCodeException() = default;
            int getResponseCode() const {
                return code_;
            }

        private:
            int code_;
        };

        struct TokenCheckResult {
            bool owner{false};
            bool guest{false};
        };

        virtual ~IBackendApi() = default;

        virtual DevicesMap getConnectedDevicesList() = 0;
        virtual std::string getToken(const DeviceId& deviceId) = 0;
        virtual bool checkToken(const std::string& token) = 0;
        virtual TokenCheckResult checkToken2(const std::string& token) = 0;
        // method to notify this class if it cache something
        virtual void invalidCert(const DeviceId&) = 0;
        virtual void invalidToken(const DeviceId&) = 0;
    };

} // namespace glagol
