#include "gogol_generator.h"

#include <chrono>

namespace {

    std::string playerStateToString(quasar::gogol::GogolGenerator::PlayerAlive::State state) {
        using State = quasar::gogol::GogolGenerator::PlayerAlive::State;
        switch (state) {
            case State::PLAY: {
                return "play";
            }
            case State::PAUSE: {
                return "pause";
            }
            case State::BUFFERING: {
                return "buffering";
            }
            case State::END: {
                return "end";
            }
        }
        // should not happen
        return "unknown";
    }

} // namespace

namespace quasar::gogol {

    Json::Value GogolGenerator::makeDefaultEvent(const Context& ctx) const {
        using namespace std::chrono;
        Json::Value json;
        json["eventType"] = EventType::EVENT;

        json["device"]["id"] = ctx.deviceId;
        json["service"] = playerName_;
        json["streamUrl"] = ctx.url;
        json["vsid"] = ctx.vsid;
        json["labels"]["from"] = service_;
        json["version"] = ctx.version;

        json["stalledCount"] = ctx.stalledCount;
        json["stalledTime"] = makeFloatTimeSec(ctx.stalledTimeSpent);

        json["watchedTime"] = makeFloatTimeSec(ctx.userTimeSpent);
        json["data"]["watchedSec"] = makeFloatTimeSec(ctx.userTimeSpent);
        json["data"]["time"] = makeFloatTimeSec(ctx.trackProgress);

        /* current utc time in ms */
        const auto now = system_clock::now();
        json["timestamp"] = int64_t(duration_cast<milliseconds>(now.time_since_epoch()).count());

        if (ctx.trackDuration.has_value()) {
            json["data"]["duration"] = makeFloatTimeSec(*ctx.trackDuration);
        }

        json["additionalParameters"]["platform"] = ctx.deviceType;

        // FIXME: verify this
        json["engine"] = "HLS";
        json["isMuted"] = false;

        return json;
    }

    Json::Value GogolGenerator::makeCreatePlayer(const Context& ctx) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventName"] = "CreatePlayer";

        return json;
    }

    Json::Value GogolGenerator::makeDestroyPlayer(const Context& ctx, std::string reason) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventName"] = "DestroyPlayer";
        json["data"]["reason"] = reason; // why player is destroyed
        return json;
    }

    Json::Value GogolGenerator::makeStart(const Context& ctx) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventName"] = "Start";
        return json;
    }

    Json::Value GogolGenerator::makeEnd(const Context& ctx) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventName"] = "End";
        return json;
    }

    Json::Value GogolGenerator::makeCustomError(const Context& ctx, std::string errorName, std::string errorMessage) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventType"] = EventType::ERROR;
        json["eventName"] = errorName;
        auto& data = json["data"];
        data["isFatal"] = false; // todo: forward fatal errors too
        data["message"] = errorMessage;
        return json;
    }

    Json::Value GogolGenerator::make4SecWatched(const Context& ctx) const {
        return makeNSecWatched(ctx, std::chrono::seconds(4));
    }

    Json::Value GogolGenerator::make10SecWatched(const Context& ctx) const {
        return makeNSecWatched(ctx, std::chrono::seconds(10));
    }

    Json::Value GogolGenerator::make30SecHeartbeat(const Context& ctx) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventName"] = "30SecHeartbeat";
        return json;
    }

    Json::Value GogolGenerator::makeNSecWatched(const Context& ctx, std::chrono::seconds sec) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventName"] = std::to_string(sec.count()) + "SecWatched";
        return json;
    }

    Json::Value GogolGenerator::makeStalled(const Context& ctx, std::chrono::milliseconds stalledDuration, int stalledId) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventName"] = "Stalled";

        json["labels"]["reason"] = "Recover";
        // stalled category: like stalled_N (N - sec)
        const auto durationSec = std::chrono::duration_cast<std::chrono::seconds>(stalledDuration);
        json["labels"]["stalledDuration"] = std::to_string(durationSec.count());

        // actual stalled duration
        json["data"]["stalledDuration"] = makeFloatTimeSec(stalledDuration);
        json["data"]["stalledId"] = stalledId;
        return json;
    }

    Json::Value GogolGenerator::makeStalledEnd(const Context& ctx, std::chrono::milliseconds stalledDuration, int stalledId) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventName"] = "StalledEnd";

        json["labels"]["reason"] = "Recover";

        // float: full stalled duration sec
        json["data"]["stalledDuration"] = makeFloatTimeSec(stalledDuration);
        json["data"]["stalledId"] = stalledId;

        return json;
    }

    Json::Value GogolGenerator::makePlayerAlive(const Context& ctx, const std::vector<PlayerAlive>& playerAlive) const {
        Json::Value json = makeDefaultEvent(ctx);
        json["eventName"] = "PlayerAlive";

        {
            auto& states = json["data"]["states"];
            states = Json::arrayValue;
            for (const auto& state : playerAlive) {
                states.append(playerAliveToJson(state));
            }
        }

        json["tags"]["event_PlayerAlive"] = 1;
        json["data"]["currentStream"]["url"] = ctx.url;
        return json;
    }

    Json::Value GogolGenerator::playerAliveToJson(const PlayerAlive& state) {
        Json::Value entry;
        entry["state"] = playerStateToString(state.state);
        entry["currentTime"] = makeFloatTimeSec(state.trackProgress);
        if (state.trackDuration) {
            entry["duration"] = makeFloatTimeSec(*state.trackDuration);
        }
        entry["watchedTime"] = makeFloatTimeSec(state.userTimeSpent);
        entry["timestamp"] = int64_t(state.timestamp.count()); // utc timestamp ms

        entry["stalledCount"] = state.stalledCount;
        entry["stalledTime"] = makeFloatTimeSec(state.stalledTimeSpent);

        // stubs: these members are required. put stubs for now
        entry["remainingBufferedTime"] = 0;
        entry["connection"] = "OK";
        entry["isVisible"] = true;

        return entry;
    }

    float GogolGenerator::makeFloatTimeSec(std::chrono::milliseconds val) {
        // gogol metrics take values like 1.34sec
        return val.count() / 1000.0;
    }

} // namespace quasar::gogol
