#pragma once

#include <json/json.h>

#include <chrono>
#include <optional>
#include <string>

namespace quasar::gogol {
    class GogolGenerator {
    public:
        struct PlayerAlive {
            enum class State {
                PLAY,
                PAUSE,
                BUFFERING,
                END
            };
            std::chrono::milliseconds trackProgress;
            std::optional<std::chrono::milliseconds> trackDuration;
            std::chrono::milliseconds timestamp; // current client timestamp
            int stalledCount{0};
            std::chrono::milliseconds stalledTimeSpent;
            std::chrono::milliseconds userTimeSpent;
            State state;
        };

        struct Context {
            std::string vsid;
            std::string url;
            std::string deviceId;
            std::string deviceType;
            std::string version;
            std::optional<std::chrono::milliseconds> trackDuration;
            std::chrono::milliseconds trackProgress{0};
            std::chrono::milliseconds userTimeSpent{0};
            std::chrono::milliseconds stalledTimeSpent{0};
            int stalledCount = 0;
        };

    public:
        GogolGenerator() = default;

        Json::Value makeCreatePlayer(const Context& ctx) const;
        Json::Value makeDestroyPlayer(const Context& ctx, std::string reason) const;
        Json::Value makeStart(const Context& ctx) const;
        Json::Value makeEnd(const Context& ctx) const;
        Json::Value makeCustomError(const Context& ctx, std::string errorName, std::string errorMessage) const;

        // heartbeats
        Json::Value make4SecWatched(const Context& ctx) const;
        Json::Value make10SecWatched(const Context& ctx) const;
        Json::Value make30SecHeartbeat(const Context& ctx) const;

        // player alive
        Json::Value makePlayerAlive(const Context& ctx, const std::vector<PlayerAlive>& playerAlive) const;

        // stalled
        Json::Value makeStalled(const Context& ctx, std::chrono::milliseconds stalledDuration, int32_t stalledId) const;
        Json::Value makeStalledEnd(const Context& ctx, std::chrono::milliseconds stalledDuration, int32_t stalledId) const;

    public:
        struct EventType {
            static constexpr const char* EVENT = "event";
            static constexpr const char* FATAL = "fatal";
            static constexpr const char* ERROR = "error";
        };

    private:
        static Json::Value playerAliveToJson(const PlayerAlive& state);
        static float makeFloatTimeSec(std::chrono::milliseconds val);
        Json::Value makeDefaultEvent(const Context& ctx) const;
        Json::Value makeNSecWatched(const Context& ctx, std::chrono::seconds sec) const;

        const std::string service_ = "SmartDevices"; // service id
        const std::string playerName_ = "YandexIOAudioPlayer";
    };

} // namespace quasar::gogol
