#include "gogol_metrics_sender.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("gogol");

namespace quasar::gogol {

    GogolMetricsSender::GogolMetricsSender(std::shared_ptr<YandexIO::IDevice> device, int queueSize, bool keepAlive)
        : defaultQueueSize_(queueSize)
        , queueSize_(queueSize)
        , httpClient_("gogol", device)
    {
        Y_VERIFY(defaultQueueSize_ > 0);
        worker_.setMaxSize(queueSize_);

        httpClient_.setTimeout(std::chrono::seconds(5));

        setKeepAlive(keepAlive);
    }

    GogolMetricsSender::~GogolMetricsSender() {
        stopped_.store(true);
    }

    void GogolMetricsSender::setKeepAlive(bool keepAlive) {
        httpClient_.setReuse(keepAlive);
    }

    void GogolMetricsSender::sendMetric(Json::Value json) {
        worker_.tryAdd([this, json{std::move(json)}]() {
            const auto jsonStr = jsonToString(json);
            YIO_LOG_DEBUG("Send gogol metric: " << json["eventName"].asString() << ". Payload: " << jsonStr);
            const std::string url = "https://log.strm.yandex.ru/log";
            // never give up to send metrics
            while (!stopped_.load()) {
                try {
                    const auto response = httpClient_.post("log", url, jsonStr);
                    YIO_LOG_DEBUG("gogol response code: " << response.responseCode << " body: " << response.body);
                    break;
                } catch (const std::exception& e) {
                    // do nothing
                }
                const auto delayBetweenNetworkErrors = std::chrono::milliseconds(1500);
                // currently we retry only network errors. There is no need for smart backoffs
                std::this_thread::sleep_for(delayBetweenNetworkErrors);
            }
        });
    }

    void GogolMetricsSender::setQueueSize(std::optional<int> sizeOpt) {
        const auto size = sizeOpt.value_or(defaultQueueSize_);
        if (size != queueSize_) {
            YIO_LOG_INFO("Update GogolMetricsSender queue size: " << size);
            queueSize_ = size;
            worker_.setMaxSize(queueSize_);
        }
    }

} // namespace quasar::gogol
