#include "gogol_session.h"

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

using namespace std::chrono;

namespace quasar::gogol {

    GogolSession::GogolSession(std::weak_ptr<IGogolMetricsSender> sender)
        : wsender_(std::move(sender))
        , worker_(std::make_shared<quasar::NamedCallbackQueue>("gogol_session"))
        , collectPlayerAliveCallback_(worker_, quasar::UniqueCallback::ReplaceType::INSERT_BACK)
    {
    }

    GogolSession::~GogolSession() {
        // dump left playerAlive metrics
        worker_->add([this]() {
            sendPlayerAlive();
        }, lifetime_);
        worker_->wait();
        lifetime_.die();
    }

    void GogolSession::setVsid(std::string vsid) {
        worker_->add([this, vsid{std::move(vsid)}]() mutable {
            context_.vsid = std::move(vsid);
        });
    }

    void GogolSession::setUrl(std::string url) {
        worker_->add([this, url{std::move(url)}]() mutable {
            context_.url = std::move(url);
        });
    }

    void GogolSession::setDeviceType(std::string deviceType) {
        worker_->add([this, deviceType{std::move(deviceType)}]() mutable {
            context_.deviceType = std::move(deviceType);
        });
    }

    void GogolSession::setDeviceId(std::string deviceId) {
        worker_->add([this, deviceId{std::move(deviceId)}]() mutable {
            context_.deviceId = std::move(deviceId);
        });
    }

    void GogolSession::setVersion(std::string version) {
        worker_->add([this, version{std::move(version)}]() mutable {
            context_.version = std::move(version);
        });
    }

    void GogolSession::sendMetric(Json::Value json) {
        if (auto sender = wsender_.lock()) {
            sender->sendMetric(std::move(json));
        }
    }

    void GogolSession::createPlayer() {
        worker_->add([this]() {
            createdTs_ = std::chrono::steady_clock::now();
            sendMetric(generator_.makeCreatePlayer(context_));

            // collect first player and schedule "collect" and "send" timers
            collectPlayerAlive();
            scheduleCollectPlayerAlive();
            // first player alive metrics should be sent in 10 sec
            // after that metric will be sent each 30 sec
            scheduleSendPlayerAlive(seconds(10));
        });
    }

    void GogolSession::destroyPlayer(std::string reason) {
        worker_->add([this, reason{std::move(reason)}]() mutable {
            sendMetric(generator_.makeDestroyPlayer(context_, std::move(reason)));
        });
    }

    void GogolSession::handleError(std::string errorName, std::string errorMessage) {
        worker_->add([this, errorName{std::move(errorName)}, errorMessage{std::move(errorMessage)}]() mutable {
            sendMetric(generator_.makeCustomError(context_, std::move(errorName), std::move(errorMessage)));
        });
    }

    void GogolSession::handleEnd() {
        worker_->add([this]() {
            collectPlayerAlvieIfStateChanged(GogolGenerator::PlayerAlive::State::END);
            sendMetric(generator_.makeEnd(context_));
        });
    }

    void GogolSession::handlePause() {
        worker_->add([this]() {
            collectPlayerAlvieIfStateChanged(GogolGenerator::PlayerAlive::State::PAUSE);
        });
    }

    void GogolSession::handleProgress(std::chrono::milliseconds pos, std::chrono::milliseconds dur) {
        worker_->add([this, pos, dur]() {
            collectPlayerAlvieIfStateChanged(GogolGenerator::PlayerAlive::State::PLAY);

            context_.trackDuration = dur;
            const auto lastPos = std::exchange(context_.trackProgress, pos);
            const auto lastUserTimeSpent = context_.userTimeSpent;

            if (!std::exchange(started_, true)) {
                sendMetric(generator_.makeStart(context_));
            }

            // handle pos after seek correctly: increase timespent only when pos increases
            if (pos >= lastPos) {
                context_.userTimeSpent += (pos - lastPos);
            }

            const auto userTimeSpentDiffSec = duration_cast<seconds>(context_.userTimeSpent) - duration_cast<seconds>(lastUserTimeSpent);
            const auto userTimeSpentSecondPassed = userTimeSpentDiffSec >= seconds(1);
            if (userTimeSpentSecondPassed) {
                const auto utsSec = duration_cast<seconds>(context_.userTimeSpent);
                if (utsSec == seconds(4)) {
                    sendMetric(generator_.make4SecWatched(context_));
                } else if (utsSec == seconds(10)) {
                    sendMetric(generator_.make10SecWatched(context_));
                } else if (utsSec.count() % 30 == 0) {
                    sendMetric(generator_.make30SecHeartbeat(context_));
                }
            }
        });
    }

    void GogolSession::handleStalled() {
        worker_->add([this]() {
            collectPlayerAlvieIfStateChanged(GogolGenerator::PlayerAlive::State::BUFFERING);
            // this method doesn't care if it will be called once a sec or more
            // it makes sure that each metric is sent once
            // but it also doesn't care if it skip one of metrics
            // so need to make sure it's called often enough
            const auto now = steady_clock::now();
            if (!stalled_.has_value()) {
                stalled_.emplace();
                // stalledId starts from zero. So take current stalledCount as ID and increase stalledCount
                stalled_->id = context_.stalledCount++;
                sendMetric(generator_.makeStalled(context_, seconds(0), stalled_->id));
                return;
            }

            const auto lastStalledDurationSec = duration_cast<seconds>(stalled_->lastTs - stalled_->startTs);
            const auto stalledDuration = duration_cast<milliseconds>(now - stalled_->startTs);
            const auto stalledDurationSec = duration_cast<seconds>(stalledDuration);

            context_.stalledTimeSpent += duration_cast<milliseconds>(now - stalled_->lastTs);
            stalled_->lastTs = now;

            // send stalledX metric when duration in sec changes from N-1 to N only (do not resend when N -> N)
            if ((stalledDurationSec - lastStalledDurationSec) >= seconds(1)) {
                if (stalledDurationSec == seconds(1) ||
                    stalledDurationSec == seconds(5) ||
                    stalledDurationSec == seconds(10)) {
                    sendMetric(generator_.makeStalled(context_, stalledDuration, stalled_->id));
                }
            }
        });
    }

    void GogolSession::handleStalledEnd() {
        worker_->add([this]() {
            if (!stalled_.has_value()) {
                return;
            }
            const auto now = std::chrono::steady_clock::now();
            context_.stalledTimeSpent += duration_cast<milliseconds>(now - stalled_->lastTs);
            const auto stalledDuration = duration_cast<milliseconds>(now - stalled_->startTs);

            sendMetric(generator_.makeStalledEnd(context_, stalledDuration, stalled_->id));

            // drop current stalled session
            stalled_.reset();
        });
    }

    void GogolSession::scheduleCollectPlayerAlive() {
        if (!createdTs_.has_value()) {
            return;
        }
        // TODO: need to reschedule collect on demand (when playerState changes)
        const auto now = steady_clock::now();
        const auto diff = duration_cast<seconds>(now - *createdTs_);
        auto delay = seconds(5); // collect playerAliveState every 5 sec by default
        if (diff < seconds(10)) {
            // first 10 sec after creation -> collect playerAliveState every second
            delay = seconds(1);
        }

        collectPlayerAliveCallback_.executeDelayed([this]() {
            collectPlayerAlive();
            scheduleCollectPlayerAlive();
        }, delay, lifetime_);
    }

    void GogolSession::collectPlayerAlive() {
        Y_ENSURE_THREAD(worker_);

        GogolGenerator::PlayerAlive alive;
        alive.state = playerState_.value_or(GogolGenerator::PlayerAlive::State::BUFFERING);
        alive.trackProgress = context_.trackProgress;
        alive.trackDuration = context_.trackDuration;
        alive.stalledCount = context_.stalledCount;
        alive.stalledTimeSpent = context_.stalledTimeSpent;
        alive.userTimeSpent = context_.userTimeSpent;
        /* current utc time */
        const auto now = system_clock::now();
        alive.timestamp = duration_cast<milliseconds>(now.time_since_epoch());

        playerAlive_.push_back(alive);
    }

    void GogolSession::sendPlayerAlive() {
        sendMetric(generator_.makePlayerAlive(context_, playerAlive_));
        playerAlive_.clear();
    }

    void GogolSession::scheduleSendPlayerAlive(std::chrono::seconds delay) {
        worker_->addDelayed([this]() {
            sendPlayerAlive();
            // send playerAlive metrics every 30 sec by default
            const auto defaultDelay = std::chrono::seconds(30);
            scheduleSendPlayerAlive(defaultDelay);
        }, delay, lifetime_);
    }

    void GogolSession::collectPlayerAlvieIfStateChanged(GogolGenerator::PlayerAlive::State newState) {
        Y_ENSURE_THREAD(worker_);
        if (!playerState_.has_value() || *playerState_ != newState) {
            playerState_ = newState;
            collectPlayerAlive();
            scheduleCollectPlayerAlive();
        }
    }

} // namespace quasar::gogol
