#pragma once

#include "gogol_generator.h"
#include "gogol_metrics_sender.h"

#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/libs/threading/lifetime.h>
#include <yandex_io/libs/threading/unique_callback.h>

#include <json/json.h>

#include <memory>
#include <string>

namespace quasar::gogol {

    class IGogolSession {
    public:
        virtual ~IGogolSession() = default;

        virtual void setVsid(std::string vsid) = 0;
        virtual void setUrl(std::string url) = 0;

        virtual void setDeviceType(std::string deviceType) = 0;
        virtual void setDeviceId(std::string deviceId) = 0;
        virtual void setVersion(std::string version) = 0;

        virtual void handleProgress(std::chrono::milliseconds /*pos*/, std::chrono::milliseconds /*dur*/) = 0;
        virtual void handlePause() = 0;
        virtual void handleEnd() = 0;
        virtual void handleError(std::string errorName, std::string errorMessage) = 0;

        virtual void handleStalled() = 0;
        virtual void handleStalledEnd() = 0;

        virtual void createPlayer() = 0;
        virtual void destroyPlayer(std::string reason) = 0;
    };

    class GogolSession: public IGogolSession {
    public:
        explicit GogolSession(std::weak_ptr<IGogolMetricsSender> sender);
        ~GogolSession();

        // TODO: can it be a constructor setting?
        void setVsid(std::string vsid) override;
        void setUrl(std::string url) override;

        void setDeviceType(std::string deviceType) override;
        void setDeviceId(std::string deviceId) override;
        void setVersion(std::string version) override;

        void handleProgress(std::chrono::milliseconds pos, std::chrono::milliseconds dur) override;
        void handlePause() override;
        void handleEnd() override;
        void handleError(std::string errorName, std::string errorMessage) override;

        void handleStalled() override;
        void handleStalledEnd() override;

        void createPlayer() override;
        void destroyPlayer(std::string reason) override;

    private:
        void sendMetric(Json::Value json);

        void collectPlayerAlive();
        void scheduleCollectPlayerAlive();

        void sendPlayerAlive();
        void collectPlayerAlvieIfStateChanged(GogolGenerator::PlayerAlive::State newState);
        void scheduleSendPlayerAlive(std::chrono::seconds delay);

        bool started_{false};

        struct StalledSession {
            int id{-1};
            std::chrono::steady_clock::time_point lastTs = std::chrono::steady_clock::now();
            const std::chrono::steady_clock::time_point startTs = std::chrono::steady_clock::now();
        };
        std::optional<StalledSession> stalled_; // current stalled session
        std::optional<std::chrono::steady_clock::time_point> createdTs_;

        std::vector<GogolGenerator::PlayerAlive> playerAlive_;

        std::optional<GogolGenerator::PlayerAlive::State> playerState_;
        GogolGenerator::Context context_;
        GogolGenerator generator_;
        const std::weak_ptr<IGogolMetricsSender> wsender_;
        std::shared_ptr<ICallbackQueue> worker_;
        UniqueCallback collectPlayerAliveCallback_;
        Lifetime lifetime_;
    };

    class NullGogolSession: public IGogolSession {
    public:
        // FIXME: to .cpp
        void setVsid(std::string /*vsid*/) override{};
        void setUrl(std::string /*url*/) override{};

        void setDeviceType(std::string /*deviceType*/) override{};
        void setDeviceId(std::string /*deviceId*/) override{};
        void setVersion(std::string /*version*/) override{};

        void handleProgress(std::chrono::milliseconds /*pos*/, std::chrono::milliseconds /*dur*/) override{};
        void handlePause() override{};
        void handleEnd() override{};
        void handleError(std::string /* errorName */, std::string /* errorMessage */) override{};

        void handleStalled() override{};
        void handleStalledEnd() override{};

        void createPlayer() override{};
        void destroyPlayer(std::string /*reason*/) override{};
    };

} // namespace quasar::gogol
