#include "http_client.h"

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/telemetry/telemetry.h>

#include <curl/curl.h>
#include <curl/easy.h>
#include <curl/multi.h>

#include <util/charset/utf8.h>
#include <util/generic/scope.h>

#include <fstream>

#include <resolv.h>

using namespace quasar;

namespace {
#if !defined(__ANDROID_API__)
    void resNinit()
    {
        static std::mutex mutex;
        std::lock_guard lock(mutex);
        res_ninit(&_res);
    }
#endif

    class CurlException: public std::runtime_error {
    public:
        CurlException(CURLcode code, const std::string& message)
            : std::runtime_error(
                  [&] {
                      if (code != CURLE_OK) {
                          return message + " [Code=" + std::to_string(code) + ": " + curl_easy_strerror(code) + "]";
                      } else {
                          return message;
                      }
                  }())
            , curlCode(code)
        {
        }
        CURLcode curlCode;
    };

} // namespace

struct HttpClient::RequestData {
    RequestData() = delete;

    RequestData(std::string_view inTag, std::string_view inVerb, const std::string& inUrl, const Headers& inHeaders, std::shared_ptr<CURL> easyPtr)
        : curlEasyPtr(easyPtr)
        , tag(inTag)
        , verb(inVerb)
        , url(inUrl)
        , headers(inHeaders)
    {
        if (!curlEasyPtr) {
            curlEasyPtr = std::shared_ptr<CURL>(curl_easy_init(), [](CURL* curlPtr) { curl_easy_cleanup(curlPtr); });
            if (!curlEasyPtr) {
                throw CurlException(CURLE_OK, "curl_easy_init() returned nullptr");
            }
        }
    }
    RequestData(const RequestData&) = delete;
    RequestData& operator=(const RequestData&) = delete;
    ~RequestData()
    {
        if (curlHeaders) {
            curl_slist_free_all(curlHeaders);
        }
    }

    std::shared_ptr<CURL> curlEasyPtr;
    char curlErrorBuffer[CURL_ERROR_SIZE];
    struct curl_slist* curlHeaders{nullptr};

    const std::string_view tag;
    const std::string_view verb;
    const std::string& url;
    const Headers& headers;

    std::string caCertsFile;

    std::function<int(char* ptr, size_t size)> headerFunction;
    std::function<int(char* ptr, size_t size)> writeFunction;

    std::exception_ptr raisedException;

    CURL* curlEasy() const {
        return curlEasyPtr.get();
    }
};

HttpClient::HttpClient(std::string name, std::shared_ptr<YandexIO::IDevice> device)
    : device_(std::move(device))
    , name_(std::move(name))
    , reportRequest_(makeDefaultMonitoring(device_->telemetry()))
{
}

void HttpClient::disableHttpRequestMetrics()
{
    reportRequest_ = nullptr;
}

void HttpClient::setCalcRetryDelayFunction(CalcRetryDelayFunction calcRetryDelayFunc)
{
    std::lock_guard lock(settingsMutex_);
    calcRetryDelayFunc_ = std::move(calcRetryDelayFunc);
}

void HttpClient::allowRequestResponseLogging(bool allow)
{
    std::lock_guard lock(settingsMutex_);
    allowRequestResponseLogging_ = allow;
}

void HttpClient::setFollowRedirect(bool followRedirect)
{
    std::lock_guard lock(settingsMutex_);
    followRedirect_ = followRedirect;
}

void HttpClient::setTimeout(std::chrono::milliseconds timeout)
{
    std::lock_guard lock(settingsMutex_);
    timeout_ = timeout;
}

void HttpClient::setConnectionTimeout(std::chrono::milliseconds timeout)
{
    std::lock_guard lock(settingsMutex_);
    connectionTimeout_ = timeout;
}

void HttpClient::setRetriesCount(int retriesCount)
{
    std::lock_guard lock(settingsMutex_);
    retriesCount_ = retriesCount;
}

void HttpClient::setMinRetriesTime(std::chrono::milliseconds minRetryTime)
{
    std::lock_guard lock(settingsMutex_);
    minRetriesTime_ = minRetryTime;
}

void HttpClient::setZeroDNSCacheTimeoutForNextRequest()
{
    std::lock_guard lock(settingsMutex_);
    zeroDnsCacheOnce_ = true;
}

void HttpClient::setDnsCacheTimeout(std::chrono::seconds timeout)
{
    std::lock_guard lock(settingsMutex_);
    dnsTimeout_ = timeout;
}

void HttpClient::cancelRetries()
{
    std::lock_guard lock(cancelRetriesMutex_);
    cancelRetries_ = true;
    cancelRetriesCV_.notify_all();
}

void HttpClient::setCorruptedBlockSize(int64_t size)
{
    std::lock_guard lock(settingsMutex_);
    corruptedBlockSize_ = size;
}

HttpClient::HttpResponse HttpClient::get(std::string_view tag, const std::string& url, const Headers& headers)
{
    return doRequest([&](HttpResponse& httpResponse) {
        auto requestData = createRequestData(tag, "GET", url, headers);
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_POST, 0L);
        setHeaderCallback(*requestData, httpResponse);
        setWriteCallback(*requestData, httpResponse);
        return std::move(requestData);
    });
}

HttpClient::HttpResponse HttpClient::head(std::string_view tag, const std::string& url, const Headers& headers)
{
    return doRequest([&](HttpResponse& httpResponse) {
        auto requestData = createRequestData(tag, "HEAD", url, headers);
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_NOBODY, 1L);
        setHeaderCallback(*requestData, httpResponse);
        return std::move(requestData);
    });
}

HttpClient::HttpResponse HttpClient::post(std::string_view tag, const std::string& url, const std::string& data, const Headers& headers)
{
    return doRequest([&](HttpResponse& httpResponse) {
        auto requestData = createRequestData(tag, "POST", url, headers);
        if (data.length() <= size_t(1 << 31)) {
            curl_easy_setopt(requestData->curlEasy(), CURLOPT_POSTFIELDSIZE, data.length());
        } else {
            curl_easy_setopt(requestData->curlEasy(), CURLOPT_POSTFIELDSIZE_LARGE, data.length());
        }
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_COPYPOSTFIELDS, data.c_str());

        setHeaderCallback(*requestData, httpResponse);
        setWriteCallback(*requestData, httpResponse);
        return requestData;
    });
}

uint32_t HttpClient::download(
    std::string_view tag,
    const std::string& url,
    const std::string& outputFileName,
    const Headers& headers,
    ProgressFunction onProgress,
    long lowSpeedLimitByteSec,
    long lowSpeedLimitTimeoutSec)
{
    std::vector<DownloadFileTask> downloadFileTasks = {
        DownloadFileTask{
            .url = url,
            .outputFileName = outputFileName,
            .headers = headers,
            .onProgress = onProgress,
            .speedLimitByteSec = lowSpeedLimitByteSec,
            .speedLimitTimeoutSec = lowSpeedLimitTimeoutSec,
        },
    };
    std::vector<uint32_t> checksums = download(tag, downloadFileTasks);
    return checksums.front();
}

std::vector<uint32_t> HttpClient::download(std::string_view tag, std::vector<DownloadFileTask>& downloadFileTasks)
{
    std::unique_lock lock(settingsMutex_);
    auto corruptedBlockSize = corruptedBlockSize_;
    lock.unlock();

    std::vector<uint32_t> result;
    result.reserve(downloadFileTasks.size());
    cancelDownload_ = false;

    CURLM* curlMulti = curl_multi_init();
    if (!curlMulti) {
        throw std::runtime_error("Fail to init curl multiple handle");
    }
    Y_DEFER {
        curl_multi_cleanup(curlMulti);
        curlMulti = nullptr;
    };

    std::vector<std::shared_ptr<RequestData>> requestDatas;
    requestDatas.reserve(downloadFileTasks.size());

    Y_DEFER {
        // Removing an easy handle from a multi handle again is completely
        // safe and returns CURLE_OK
        for (auto& rd : requestDatas) {
            curl_multi_remove_handle(curlMulti, rd->curlEasy());
        }
    };

    std::vector<HttpResponse> httpResponses;
    httpResponses.reserve(downloadFileTasks.size());

    std::vector<std::fstream> files;
    files.reserve(downloadFileTasks.size());

    std::vector<int64_t> filesizes;
    filesizes.reserve(downloadFileTasks.size());

    /*
     * Prepare all tasks for fetching
     */
    for (size_t taskIndex = 0; taskIndex < downloadFileTasks.size(); ++taskIndex) {
        auto& task = downloadFileTasks[taskIndex];
        auto& httpResponse = httpResponses.emplace_back();
        auto& requestData = requestDatas.emplace_back(createRequestData(tag, "GET", task.url, task.headers));
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_POST, 0L);
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_WRITEFUNCTION, writeCallback);
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_WRITEDATA, requestData.get());

        const auto& filename = task.outputFileName;
        truncateCorruptedBlock(filename, corruptedBlockSize);
        auto& file = files.emplace_back();
        file.open(filename, std::ofstream::binary | std::ofstream::out | std::ofstream::app | std::ofstream::ate);
        if (!file.good()) {
            throw std::runtime_error("Cannot open " + filename + " for writing.");
        }
        auto filesize = filesizes.emplace_back(file.tellg());
        if (filesize) {
            requestData->curlHeaders = curl_slist_append(requestData->curlHeaders, ("Range: bytes=" + std::to_string(filesize) + "-").c_str());
            curl_easy_setopt(requestData->curlEasy(), CURLOPT_HTTPHEADER, requestData->curlHeaders);
        }
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_LOW_SPEED_LIMIT, task.speedLimitByteSec);
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_LOW_SPEED_TIME, task.speedLimitTimeoutSec);

        setHeaderCallback(*requestData, httpResponse);
        requestData->writeFunction =
            [this, &downloadFileTasks, &requestDatas, &httpResponses, &files, filesize, taskIndex](char* ptr, size_t size) -> int {
            if (cancelDownload_) {
                return 0;
            }
            processHeaders(*requestDatas[taskIndex], httpResponses[taskIndex]);
            auto& file = files[taskIndex];
            if (file.is_open()) {
                file.write(ptr, size);

                if (auto& progress = downloadFileTasks[taskIndex].onProgress) {
                    progress(file.tellg(), filesize + httpResponses[taskIndex].contentLength);
                }
            }
            return size;
        };

        curl_multi_add_handle(curlMulti, requestData->curlEasy());
    }

    /*
     * Fetch all tasks
     */
    int isRunning{0};
    int unused{0};
    do {
        CURLMcode mcode = curl_multi_perform(curlMulti, &isRunning);
        for (auto& rd : requestDatas) {
            if (rd->raisedException) {
                std::rethrow_exception(rd->raisedException);
            }
        }
        if (mcode == CURLM_OK) {
            constexpr int TIMEOUT_MS{100};
            mcode = curl_multi_poll(curlMulti, nullptr, 0, TIMEOUT_MS, &unused);
        }

        if (cancelDownload_) {
            throw std::runtime_error("Download canceled");
        }

        if (mcode != CURLM_OK) {
            YIO_LOG_WARN("HTTP{" << name_ << ", " << tag << "} download of all tasks failed");
            throw std::runtime_error("Fail to download any files, curl return fatal error");
        }

        while (CURLMsg* msg = curl_multi_info_read(curlMulti, &unused)) {
            if (msg->msg == CURLMSG_DONE) {
                if (msg->data.result != CURLE_OK) {
                    for (auto& rd : requestDatas) {
                        if (rd->curlEasy() == msg->easy_handle) {
                            throw CurlException(msg->data.result, "Fail to download url=" + rd->url);
                        }
                    }
                }
                curl_multi_remove_handle(curlMulti, msg->easy_handle);
            }
        }
    } while (isRunning);

    /*
     * Finalize all
     */
    for (size_t taskIndex = 0; taskIndex < downloadFileTasks.size(); ++taskIndex) {
        auto& task = downloadFileTasks[taskIndex];
        auto& httpResponse = httpResponses[taskIndex];
        auto& file = files[taskIndex];
        auto filesize = filesizes[taskIndex];

        if (httpResponse.responseCode < 200 || httpResponse.responseCode >= 300) {
            std::stringstream ss;
            ss << "Invalid response code: " << httpResponse.responseCode << ", body: " << httpResponse.body << ". Download url=" << task.url << " failed";
            throw CurlException(CURLE_OK, ss.str());
        }
        if (file.tellg() != filesize + httpResponse.contentLength) {
            throw CurlException(CURLE_OK, "File \"" + task.outputFileName + "\" size mismatch. Expected " + std::to_string(httpResponse.contentLength) + " but have " + std::to_string(file.tellg()));
        }
        file.close();
        quasar::Crc32 crc32;
        crc32.processFile(task.outputFileName);
        result.push_back(crc32.checksum());
        task.responseHeaders = std::move(httpResponse.extraHeaders);
    }
    return result;
}

void HttpClient::cancelDownload()
{
    cancelDownload_ = true;
}

std::shared_ptr<HttpClient::RequestData> HttpClient::createRequestData(std::string_view tag, std::string_view verb, const std::string& url, const Headers& headers)
{
    std::unique_lock lock(settingsMutex_);

    const auto& common = device_->configuration()->getServiceConfig("common");
    const auto verbose = tryGetBool(common, "curlVerbose", false);

    std::shared_ptr<RequestData> requestData = std::make_shared<RequestData>(tag, verb, url, headers, (reuse_ && requestData_) ? requestData_->curlEasyPtr : nullptr);
    requestData->caCertsFile = tryGetString(common, "caCertsFile");

    curl_easy_setopt(requestData->curlEasy(), CURLOPT_NOSIGNAL, 1);
    curl_easy_setopt(requestData->curlEasy(), CURLOPT_VERBOSE, (verbose ? 1 : 0));
    curl_easy_setopt(requestData->curlEasy(), CURLOPT_ERRORBUFFER, requestData->curlErrorBuffer);
    curl_easy_setopt(requestData->curlEasy(), CURLOPT_TIMEOUT_MS, timeout_.count());
    curl_easy_setopt(requestData->curlEasy(), CURLOPT_CONNECTTIMEOUT_MS, connectionTimeout_.count());
    curl_easy_setopt(requestData->curlEasy(), CURLOPT_FOLLOWLOCATION, (followRedirect_ ? 1 : 0));
    curl_easy_setopt(requestData->curlEasy(), CURLOPT_URL, url.data());

    if (!requestData->caCertsFile.empty()) {
        auto result = curl_easy_setopt(requestData->curlEasy(), CURLOPT_CAINFO, requestData->caCertsFile.c_str());
        if (result != CURLE_OK) {
            throw CurlException(result, "Can not set CAINFO to path '" + requestData->caCertsFile + "'");
        }
    }

    for (const auto& header : headers) {
        requestData->curlHeaders = curl_slist_append(requestData->curlHeaders, (header.first + ": " + header.second).c_str());
    }
    requestData->curlHeaders = curl_slist_append(requestData->curlHeaders, ("User-Agent: " + device_->getDeprecatedUserAgent()).c_str());
    curl_easy_setopt(requestData->curlEasy(), CURLOPT_HTTPHEADER, requestData->curlHeaders);

    curl_easy_setopt(requestData->curlEasy(), CURLOPT_LOW_SPEED_LIMIT, 0);
    curl_easy_setopt(requestData->curlEasy(), CURLOPT_LOW_SPEED_TIME, 0);

    auto dnsTimeout = dnsTimeout_;
    if (std::exchange(zeroDnsCacheOnce_, false)) {
        dnsTimeout = std::chrono::seconds(0);
    }
    curl_easy_setopt(requestData->curlEasy(), CURLOPT_DNS_CACHE_TIMEOUT, dnsTimeout.count());
    if (reuse_) {
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_MAXAGE_CONN, 180L);
        curl_easy_setopt(requestData->curlEasy(), CURLOPT_FORBID_REUSE, 0L);
        requestData_ = requestData;
    }
    return requestData;
}

void HttpClient::setWriteCallback(RequestData& requestData, HttpResponse& httpResponse)
{
    curl_easy_setopt(requestData.curlEasy(), CURLOPT_WRITEFUNCTION, writeCallback);
    curl_easy_setopt(requestData.curlEasy(), CURLOPT_WRITEDATA, &requestData);
    requestData.writeFunction =
        [&requestData, &httpResponse](char* ptr, size_t size) -> int {
        processHeaders(requestData, httpResponse);
        if (httpResponse.responseCode < 0) {
            throw CurlException(CURLE_OK, "Response code is undefined");
        }
        if (httpResponse.contentLength > 0) {
            httpResponse.body.reserve(httpResponse.contentLength);
        }
        httpResponse.body += std::string(ptr, size);
        return size;
    };
}

void HttpClient::setHeaderCallback(RequestData& requestData, HttpResponse& httpResponse)
{
    curl_easy_setopt(requestData.curlEasy(), CURLOPT_HEADERFUNCTION, headerCallback);
    curl_easy_setopt(requestData.curlEasy(), CURLOPT_HEADERDATA, &requestData);
    requestData.headerFunction =
        [&httpResponse](char* ptr, size_t size) -> int {
        const char* eq = strchr(ptr, ':');
        if (eq != nullptr) {
            std::string key(ptr, eq - ptr);
            if (key.length() == size - 1)
            {
                httpResponse.extraHeaders.emplace_back(std::move(key), std::string());
                return size;
            }

            const char* end = ptr + size;
            ++eq;
            while (eq < end && ' ' == *eq) {
                ++eq;
            }

            while (end > eq && (*(end - 1) == '\r' || *(end - 1) == '\n')) {
                --end;
            }

            std::string value(eq, end - eq);
            httpResponse.extraHeaders.emplace_back(std::move(key), std::move(value));
        }
        return size;
    };
}

HttpClient::HttpResponse HttpClient::doRequest(const std::function<std::shared_ptr<RequestData>(HttpResponse&)>& requestDataCreator)
{
    std::unique_lock lock(settingsMutex_);
    auto retryCount = retriesCount_ + 1;
    auto retryUntil = std::chrono::steady_clock::now() + std::chrono::milliseconds(minRetriesTime_);
    lock.unlock();

    int retry = 1;
    std::shared_ptr<RequestData> requestData;
    CURLcode exceptionCurlCode{CURLE_OK};

    std::unique_lock cancelRetriesLock(cancelRetriesMutex_);
    cancelRetries_ = false;
    cancelRetriesLock.unlock();

    bool reportedSslError = false;
    for (; retry <= retryCount || std::chrono::steady_clock::now() < retryUntil; ++retry) {
        try {
            exceptionCurlCode = CURLE_OK;
            HttpResponse httpResponse;
            requestData = requestDataCreator(httpResponse);
            if (retry == 1) {
                YIO_LOG_DEBUG("HTTP{" << name_ << ", " << requestData->tag << "} Request url=" << requestData->url);
            } else {
                YIO_LOG_DEBUG("HTTP{" << name_ << ", " << requestData->tag << "} Retry request failed (" << retry << " out of " << retryCount << ")");
            }
            const auto performBegin = std::chrono::steady_clock::now();
            const auto code = curl_easy_perform(requestData->curlEasy());
            const auto performEnd = std::chrono::steady_clock::now();

            // retrieve data from headers in case if writeFunction wasn't call (i.e. empty body/ head request)
            processHeaders(*requestData, httpResponse);

            if (reportRequest_) {
                const auto diff = std::chrono::duration_cast<std::chrono::milliseconds>(performEnd - performBegin);
                std::optional<int> responseCode;
                if (code == CURLE_OK) {
                    responseCode = httpResponse.responseCode;
                } else {
                    responseCode = 6000 + code;
                }
                reportRequest_(name_, requestData->verb, requestData->tag, responseCode, diff);
            }
            if (requestData->raisedException) {
                std::rethrow_exception(requestData->raisedException);
            }
            if (code != CURLE_OK) {
                if (code == CURLE_SSL_CACERT) {
                    if (!reportedSslError) {
                        Json::Value payload;
                        payload["client_name"] = name_;
                        payload["tag"] = std::string(requestData->tag);
                        device_->telemetry()->reportEvent("http_request_ssl_error", jsonToString(payload));
                        reportedSslError = true;
                    }
                    throw CurlException(code, "SSL error: will try to re-sync time");
                }
#if !defined(__ANDROID_API__)
                else if (code == CURLE_COULDNT_RESOLVE_HOST) {
                    resNinit();
                    throw CurlException(code, "DNS error: will try to reload resolv.conf");
                }
#endif
                else {
                    throw CurlException(code, "Fail to perform url=" + requestData->url);
                }
            } else {
                YIO_LOG_DEBUG("HTTP{" << name_ << ", " << requestData->tag << "} Success url=" << requestData->url);
            }

            if (allowRequestResponseLogging_) {
                if (IsUtf(httpResponse.body)) {
                    YIO_LOG_DEBUG("HTTP{" << name_ << ", " << requestData->tag << "} Got response: retries=" << retry << " code=" << httpResponse.responseCode << " -->> " << httpResponse.body);
                } else {
                    YIO_LOG_DEBUG("HTTP{" << name_ << ", " << requestData->tag << "} Got response: retries=" << retry << " code=" << httpResponse.responseCode << " -->> <non utf body omitted>");
                }
            }
            return httpResponse;
        } catch (const CurlException& ce) {
            exceptionCurlCode = ce.curlCode;
            YIO_LOG_WARN("HTTP{" << name_ << ", " << requestData->tag << "} " << ce.what() << " Attempts left: " << retryCount - retry);
        } catch (const std::exception& ex) {
            YIO_LOG_WARN("HTTP{" << name_ << ", " << requestData->tag << "} " << ex.what() << " Attempts left: " << retryCount - retry);
        }

        lock.lock();
        constexpr std::chrono::milliseconds retryDelay{500};
        const auto sleepTime = (calcRetryDelayFunc_ ? calcRetryDelayFunc_(retry - 1) : retryDelay);
        lock.unlock();

        if (Y_UNLIKELY(retry >= retryCount && std::chrono::steady_clock::now() + sleepTime >= retryUntil)) {
            break;
        }

        YIO_LOG_DEBUG("Sleep for " << sleepTime.count() << " ms before next HTTP{" << name_ << ", " << requestData->tag << "} retry...");
        cancelRetriesLock.lock();
        cancelRetriesCV_.wait_for(cancelRetriesLock, sleepTime, [this] { return cancelRetries_; });
        if (cancelRetries_) {
            YIO_LOG_DEBUG("HTTP{" << name_ << ", " << requestData->tag << "} request retries are interrupted");
            break;
        }
        cancelRetriesLock.unlock();
    }

    std::string_view tag = (requestData ? requestData->tag : "<unknown tag>");
    std::string url = (requestData ? requestData->url : "<unknwon url>");
    YIO_LOG_WARN("HTTP{" << name_ << ", " << tag << "} request (" << url << ") failed after " << retry - 1 << " attempts");
    throw CurlException(exceptionCurlCode, "Can not get '" + url + "'");
}

void HttpClient::processHeaders(RequestData& requestData, HttpResponse& httpResponse)
{
    if (httpResponse.responseCode < 0) {
        CURLcode result{CURLE_OK};

        // Read response code
        long responseCode = 0;
        result = curl_easy_getinfo(requestData.curlEasy(), CURLINFO_RESPONSE_CODE, &responseCode);
        if (result != CURLE_OK) {
            throw CurlException(result, "Fail to read response code");
        }
        httpResponse.responseCode = (int)responseCode;

        // Read content length
        double contentLength;
        result = curl_easy_getinfo(requestData.curlEasy(), CURLINFO_CONTENT_LENGTH_DOWNLOAD, &contentLength);
        if (result != CURLE_OK) {
            throw CurlException(result, "Fail to read content length");
        }
        httpResponse.contentLength = (int64_t)contentLength;

        // Read content type
        char* contentType = nullptr;
        result = curl_easy_getinfo(requestData.curlEasy(), CURLINFO_CONTENT_TYPE, &contentType);
        if (result != CURLE_OK)
        {
            throw CurlException(result, "Fail to read content type");
        }
        if (contentType) {
            httpResponse.contentType = contentType;
        }
    }
}

size_t HttpClient::writeCallback(char* ptr, size_t size, size_t nmemb, void* requestData) noexcept {
    RequestData* rd = reinterpret_cast<RequestData*>(requestData);
    try {
        if (rd->writeFunction) {
            return rd->writeFunction(ptr, size * nmemb);
        }
    } catch (...) {
        rd->raisedException = std::current_exception();
    }
    return 0;
}

size_t HttpClient::headerCallback(char* ptr, size_t size, size_t nmemb, void* requestData) noexcept {
    RequestData* rd = reinterpret_cast<RequestData*>(requestData);
    try {
        if (rd->headerFunction) {
            return rd->headerFunction(ptr, size * nmemb);
        }
    } catch (...) {
        rd->raisedException = std::current_exception();
        return 0;
    }
    return size * nmemb;
}

void HttpClient::truncateCorruptedBlock(const std::string& filename, int64_t corruptedBlockSize)
{
    if (fileExists(filename)) {
        int64_t fileSize = getFileSize(filename);
        if (fileSize < corruptedBlockSize) {
            YIO_LOG_INFO("Removing file " << filename << " of size " << fileSize);
            std::remove(filename.c_str());
        } else {
            auto stubSize = fileSize - corruptedBlockSize;
            YIO_LOG_INFO("Truncating file " << filename << " size from " << fileSize << " to " << stubSize);
            std::ignore = ::truncate(filename.c_str(), stubSize);
        }
    }
}

void HttpClient::addSignatureHeaders(Headers& headers, const std::string& signature, const std::string& cryptographyType, int version) {
    // Now we use proper RSA signatures in all DeviceCryptography implementations
    // See https://st.yandex-team.ru/QUASARINFRA-147
    headers["X-Quasar-Signature-Version"] = std::to_string(version);
    headers["X-Quasar-Signature-Cryptography"] = cryptographyType;
    headers["X-Quasar-Signature"] = urlEncode(base64Encode(signature.c_str(), signature.size()));
}

HttpClient::ReportRequestMetrica HttpClient::makeDefaultMonitoring(std::shared_ptr<YandexIO::ITelemetry> telemetry) {
    Y_VERIFY(telemetry);

    return [telemetry{std::move(telemetry)}](auto name, auto verb, auto tag, auto code, auto timings) {
        Json::Value payload;
        payload["method"] = std::string(verb);
        payload["tag"] = std::string(tag);
        payload["client_name"] = std::string(name);
        payload["code"] = code.value_or(0); // 0 is timeout
        payload["timing_ms"] = int64_t(timings.count());
        telemetry->reportEvent("http_request", quasar::jsonToString(payload));
    };
}

void HttpClient::globalInit()
{
    CURLcode rv = curl_global_init(CURL_GLOBAL_ALL);
    if (rv != CURLE_OK) {
        std::stringstream ss;
        ss << "curl global init failed (error code: " << rv << ')';
        throw std::runtime_error(ss.str());
    }
}

void HttpClient::setReuse(bool reuse) {
    reuse_ = reuse;
}
