#pragma once
#include "i_http_client.h"

#include <yandex_io/libs/device/i_device.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <atomic>

namespace quasar {

    class HttpClient: public IHttpClient {
    public:
        /**
         * @brief Init curl using: curl_global_init. This function IS NOT THREAD SAFE. According to curl_global_init documentation
         * this function should be called before any other thread started.
         */
        static void globalInit();

        using CalcRetryDelayFunction = std::function<std::chrono::milliseconds(const int retryNum)>;
        using ReportRequestMetrica = std::function<void(std::string_view name, std::string_view verb, std::string_view tag,
                                                        std::optional<int> code, std::chrono::milliseconds timings)>;

        static ReportRequestMetrica makeDefaultMonitoring(std::shared_ptr<YandexIO::ITelemetry> telemetry);

        HttpClient(std::string name, std::shared_ptr<YandexIO::IDevice> device);
        HttpClient(const HttpClient&) = delete;
        HttpClient& operator=(const HttpClient&) = delete;

        void disableHttpRequestMetrics();
        void setCalcRetryDelayFunction(CalcRetryDelayFunction calcRetryDelayFunc);
        void allowRequestResponseLogging(bool allow);

        HttpResponse get(std::string_view tag, const std::string& url, const Headers& headers = {}) override;
        HttpResponse head(std::string_view tag, const std::string& url, const Headers& headers = {}) override;
        HttpResponse post(std::string_view tag, const std::string& url, const std::string& data, const Headers& headers = {}) override;

        std::vector<uint32_t> download(std::string_view tag, std::vector<DownloadFileTask>& downloadFileTasks) override;
        uint32_t download(std::string_view tag, const std::string& url, const std::string& outputFileName,
                          const Headers& headers = {}, ProgressFunction onProgress = ProgressFunction(),
                          long lowSpeedLimitByteSec = 0, long lowSpeedLimitTimeoutSec = 0) override;
        void cancelDownload() override;

        void setFollowRedirect(bool followRedirect);
        void setTimeout(std::chrono::milliseconds timeout) override;
        void setConnectionTimeout(std::chrono::milliseconds timeout);
        void setRetriesCount(int retriesCount);
        void setMinRetriesTime(std::chrono::milliseconds minRetryTime);
        void setZeroDNSCacheTimeoutForNextRequest();
        void setDnsCacheTimeout(std::chrono::seconds timeout);

        void cancelRetries();
        void setCorruptedBlockSize(int64_t size);

        void setReuse(bool reuse);

        static void addSignatureHeaders(Headers& headers, const std::string& signature, const std::string& cryptographyType, int version = 2);

    private:
        struct RequestData;
        std::shared_ptr<RequestData> createRequestData(std::string_view tag, std::string_view verb, const std::string& url, const Headers& headers);
        static void setWriteCallback(RequestData& requestData, HttpResponse& httpResponse);
        static void setHeaderCallback(RequestData& requestData, HttpResponse& httpResponse);

        HttpResponse doRequest(const std::function<std::shared_ptr<RequestData>(HttpResponse&)>& /*requestDataCreator*/);
        static void processHeaders(RequestData& requestData, HttpResponse& httpResponse);
        static size_t writeCallback(char* ptr, size_t size, size_t nmemb, void* requestData) noexcept;
        static size_t headerCallback(char* ptr, size_t size, size_t nmemb, void* requestData) noexcept;
        static void truncateCorruptedBlock(const std::string& filename, int64_t corruptedBlockSize);

    private:
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::string name_;
        ReportRequestMetrica reportRequest_;
        std::shared_ptr<RequestData> requestData_;

        bool zeroDnsCacheOnce_{false};
        bool reuse_{false};
        std::atomic<bool> cancelDownload_{false};

        std::mutex cancelRetriesMutex_;
        bool cancelRetries_{false};
        quasar::SteadyConditionVariable cancelRetriesCV_;

        std::mutex settingsMutex_;
        CalcRetryDelayFunction calcRetryDelayFunc_;
        bool allowRequestResponseLogging_{true};
        bool followRedirect_{false};
        std::chrono::milliseconds timeout_{500};
        std::chrono::milliseconds connectionTimeout_{0};
        int retriesCount_{0};
        std::chrono::milliseconds minRetriesTime_{0};
        int64_t corruptedBlockSize_{0};

        static constexpr std::chrono::seconds DEFAULT_DNS_TIMEOUT{60};
        std::chrono::seconds dnsTimeout_{DEFAULT_DNS_TIMEOUT};
    };

} // namespace quasar
