#pragma once

#include <yandex_io/libs/http_client/i_simple_http_client.h>
#include <yandex_io/libs/device/i_device.h>
#include <yandex_io/libs/telemetry/telemetry.h>

#include <chrono>
#include <map>

namespace quasar {

    class IHttpClient: public ISimpleHttpClient {
    public:
        using Header = ISimpleHttpClient::Header;
        using Headers = ISimpleHttpClient::Headers;
        using HttpResponse = ISimpleHttpClient::HttpResponse;
        using ProgressFunction = std::function<void(int64_t downloaded, int64_t total)>;

        // Download file task for parallel downloads
        struct DownloadFileTask {
            std::string url;
            std::string outputFileName;
            Headers headers;
            std::vector<Header> responseHeaders;
            ProgressFunction onProgress;
            int64_t speedLimitByteSec{0};
            int64_t speedLimitTimeoutSec{0};

            bool operator==(const DownloadFileTask& other) const;
            bool operator<(const DownloadFileTask& other) const;
        };

        // @brief Download multiple files simultaneously, with DownloadFileTask for each one.
        // Should scale ok up to 10-s of files.
        // @see HttpClientDatacratic::download for field explanations
        virtual std::vector<uint32_t> download(std::string_view tag, std::vector<DownloadFileTask>& downloadFileTasks) = 0;

        // @brief Download content using the specified url and write it to outputFileName.
        // In case if file already exists it truncate last corruptedBlockSize_ bytes and continue downloading.
        // @see setCorruptedBlockSize
        // @param tag - logging tag like in get()/head()/post()
        // @param url - url to download from
        // @param outputFileName - file to write content to
        // @param headers - Request Headers
        // @param onProgress - function that will be called on Data Received
        // @param lowSpeedLimitByteSec - the minimum speed (byte per sec) that can be during download.
        //                               If currentSpeed < lowSpeedLimitByteSec during lowSpeedLimitTimeoutSec
        //                               exception will be thrown;
        //                               If lowSpeedLimitByteSec <= 0 or lowSpeedLimitTimeoutSec <= 0 both values will be
        //                               ignored
        // @param lowSpeedLimitTimeoutSec - Timeout Sec for low download speed.
        //                                  If currentSpeed < lowSpeedLimitByteSec during lowSpeedLimitTimeoutSec
        //                                  exception will be thrown
        // @return - Returns crc32 of downloaded file
        virtual uint32_t download(std::string_view tag, const std::string& url, const std::string& outputFileName,
                                  const Headers& headers = {}, ProgressFunction onProgress = ProgressFunction(),
                                  long lowSpeedLimitByteSec = 0, long lowSpeedLimitTimeoutSec = 0) = 0;
        virtual void cancelDownload() = 0;

        virtual void setTimeout(std::chrono::milliseconds timeout) = 0;
    };

} // namespace quasar
