#include "asio_async_object.h"
#include "asio_async_object_storage.h"

#include <yandex_io/libs/logging/logging.h>

#include <chrono>
#include <util/system/yassert.h>

using namespace quasar;
using namespace quasar::ipc::detail::asio_ipc;

std::shared_ptr<AsioAsyncObjectStorage> AsioAsyncObjectStorage::createInstance(std::shared_ptr<ICallbackQueue> gc) {
    Y_VERIFY(gc);
    return std::make_shared<AsioAsyncObjectStorage>(std::move(gc), Tag{});
}

AsioAsyncObjectStorage::AsioAsyncObjectStorage(std::shared_ptr<ICallbackQueue> gc, Tag /*unused*/)
    : gc_(std::move(gc))
{
    YIO_LOG_DEBUG("Create AsioAsyncObjectStorage " << this);
}

AsioAsyncObjectStorage::~AsioAsyncObjectStorage() {
    YIO_LOG_DEBUG("Destory AsioAsyncObjectStorage " << this);
}

std::vector<std::weak_ptr<AsioAsyncObject>> AsioAsyncObjectStorage::asyncObjects() {
    std::lock_guard lock(mutex_);
    return asyncObjects_;
}

void AsioAsyncObjectStorage::shutdownObjects() {
    YIO_LOG_INFO("Gracefully stopping ASIO async objects: " << this);

    decltype(asyncObjects_) tmpObjects;
    {
        std::lock_guard lock(mutex_);
        if (shutdown_) {
            return;
        }
        shutdown_ = true;
        std::swap(tmpObjects, asyncObjects_);
    }

    auto removeExpiredRef = [&] {
        auto end = std::remove_if(tmpObjects.begin(), tmpObjects.end(), [](const auto& ref) { return ref.expired(); });
        if (auto expiredCount = std::distance(end, tmpObjects.end())) {
            tmpObjects.resize(tmpObjects.size() - expiredCount);
        }
    };

    size_t initialAliveObjects = 0;
    for (auto& wObject : tmpObjects) {
        if (auto sObject = wObject.lock()) {
            sObject->asyncShutdown();
            ++initialAliveObjects;
        }
    }

    constexpr auto iterationQuant = std::chrono::milliseconds{25};
    constexpr auto awatingTimeout = std::chrono::seconds(2);
    const auto until = std::chrono::steady_clock::now() + awatingTimeout;
    removeExpiredRef();
    do {
        if (tmpObjects.empty()) {
            YIO_LOG_INFO("All ASIO async objects at " << this << " are stopped (max=" << maxAsyncObjectCount_ << ", unique=" << uniqueAsyncObjectCount_ << ")");
            break;
        }
        std::this_thread::sleep_for(iterationQuant);
        removeExpiredRef();
    } while (until > std::chrono::steady_clock::now());

    if (!tmpObjects.empty()) {
        YIO_LOG_WARN("Some ASIO async objects a still alive: " << tmpObjects.size() << " of " << initialAliveObjects << " (max=" << maxAsyncObjectCount_ << ", unique=" << uniqueAsyncObjectCount_ << ")");
        for (auto& wObject : tmpObjects) {
            if (auto sObject = wObject.lock()) {
                YIO_LOG_DEBUG("    " << *sObject);
            } else {
                YIO_LOG_DEBUG("    expired=" << wObject.expired());
            }
        }
    }
}

void AsioAsyncObjectStorage::markAsDirty() {
    if (gcScheduled_.exchange(true) == false) {
        constexpr auto delayedStart = std::chrono::seconds{1};
        gc_->addDelayed([this] {
            std::lock_guard lock(mutex_);
            if (shutdown_) {
                return;
            }
            gcScheduled_ = false;
            auto end = std::remove_if(asyncObjects_.begin(), asyncObjects_.end(), [](const auto& ref) { return ref.expired(); });
            auto expiredCount = std::distance(end, asyncObjects_.end());
            if (expiredCount) {
                YIO_LOG_DEBUG("Clean up " << expiredCount << " expired object(s): " << this);
                asyncObjects_.resize(asyncObjects_.size() - expiredCount);
            }
            YIO_LOG_DEBUG("AsioAsyncObjectStorage " << this << " statistic: now=" << asyncObjects_.size() << ", max=" << maxAsyncObjectCount_ << ", unique=" << uniqueAsyncObjectCount_);
        }, delayedStart, weak_from_this());
    }
}

bool AsioAsyncObjectStorage::registerObject(std::weak_ptr<AsioAsyncObject> weakObj) {
    std::lock_guard lock(mutex_);
    if (shutdown_) {
        return false;
    }
    asyncObjects_.push_back(std::move(weakObj));
    ++uniqueAsyncObjectCount_;
    maxAsyncObjectCount_ = std::max(asyncObjects_.size(), maxAsyncObjectCount_);
    return true;
}
