#pragma once

#include <yandex_io/libs/threading/i_callback_queue.h>

#include <future>
#include <memory>
#include <mutex>
#include <vector>

namespace quasar::ipc::detail::asio_ipc {

    class AsioAsyncObject;
    class AsioAsyncObjectStorage: public std::enable_shared_from_this<AsioAsyncObjectStorage> {
        struct Tag {};

    public:
        static std::shared_ptr<AsioAsyncObjectStorage> createInstance(std::shared_ptr<ICallbackQueue> gc);

        AsioAsyncObjectStorage(std::shared_ptr<ICallbackQueue> gc, Tag /*unused*/);
        ~AsioAsyncObjectStorage();

        template <class T>
        std::shared_ptr<T> registerObject(std::unique_ptr<T>&& uniqueObj) {
            auto wself = weak_from_this();
            auto sharedObj = std::shared_ptr<T>(uniqueObj.get(), [wself](T* obj) {
                delete obj;
                if (auto self = wself.lock()) {
                    self->markAsDirty();
                }
            });
            uniqueObj.release();
            return registerObject(sharedObj) ? sharedObj : nullptr;
        }

        template <class T>
        std::shared_ptr<T> registerObject(std::unique_ptr<T>&& uniqueObj, std::future<void>& deathFuture) {
            auto wself = weak_from_this();
            auto deathPromise = std::make_unique<std::promise<void>>();
            deathFuture = deathPromise->get_future();
            auto sharedObj = std::shared_ptr<T>(uniqueObj.get(), [wself, dp = deathPromise.get()](T* obj) {
                delete obj;
                if (auto self = wself.lock()) {
                    self->markAsDirty();
                }
                dp->set_value();
                delete dp;
            });
            deathPromise.release();
            uniqueObj.release();
            return registerObject(sharedObj) ? sharedObj : nullptr;
        }

        std::vector<std::weak_ptr<AsioAsyncObject>> asyncObjects();

        void shutdownObjects();

    private:
        void markAsDirty();
        bool registerObject(std::weak_ptr<AsioAsyncObject> weakObj);

    private:
        const std::shared_ptr<ICallbackQueue> gc_;
        std::atomic<bool> gcScheduled_{false};

        std::mutex mutex_;
        bool shutdown_{false};
        size_t maxAsyncObjectCount_{0};
        size_t uniqueAsyncObjectCount_{0};
        std::vector<std::weak_ptr<AsioAsyncObject>> asyncObjects_;
    };

} // namespace quasar::ipc::detail::asio_ipc
