#include "asio_async_worker.h"
#include "asio_async_object.h"
#include "asio_callback_pool.h"

#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/timer_service.h>

using namespace quasar;
using namespace quasar::ipc::detail::asio_ipc;
using namespace std::chrono_literals;

AsioAsyncWorker::~AsioAsyncWorker() {
    YIO_LOG_INFO("destroy ASIO async worker: " << this);
    Y_VERIFY(!ioThread_.joinable());
}

bool AsioAsyncWorker::isRunning() const {
    bool result = isRunning_.load();
    YIO_LOG_DEBUG("(check) ASIO worker " << this << " is " << (result ? "running" : "stopped"));
    return result;
}

void AsioAsyncWorker::startBackgroundThreads() {
    // Allow async objects to spawn new child objects
    isRunning_.store(true);

    // Keep event loop running even if there are no outstanding tasks
    asyncWorkGuard_ = std::make_shared<asio::executor_work_guard<asio::io_context::executor_type>>(ioContext_.get_executor());

    asioCallbackPool_ = std::make_shared<AsioCallbackPool>("AsioAsyncWorkerPool", 4, std::make_shared<TimerService>());

    asioAsyncObjectStorage_ = AsioAsyncObjectStorage::createInstance(asioCallbackPool_->createAsioCallbackQueue("AsioAsyncObjectStorageQueue"));

    // Start event loop thread
    ioThread_ = std::thread([this] {
        // auto work = asio::make_work_guard(this_->ioContext_);

        YIO_LOG_DEBUG("ASIO event loop begin: " << this);
        ioContext_.run();
        YIO_LOG_DEBUG("ASIO event loop end: " << this);
    });
}

void AsioAsyncWorker::stopBackgroundThreads() {
    // Prevent async objects from creating new child objects
    isRunning_.store(false);

    // Terminate event loop once all outstanding tasks are executed
    asyncWorkGuard_ = nullptr;

    asioAsyncObjectStorage_->shutdownObjects();
    asioAsyncObjectStorage_ = nullptr;

    if (asioCallbackPool_) {
        YIO_LOG_INFO("Destroying ASIO IPC worker callback queues (size=" << asioCallbackPool_->size() << "): " << this);
        asioCallbackPool_->destroy();
        asioCallbackPool_ = nullptr;
        YIO_LOG_INFO("ASIO IPC worker callback queues are destored: " << this);
    }

    YIO_LOG_INFO("Stopping ASIO IPC: " << this);
    ioContext_.stop();

    YIO_LOG_INFO("Joining ASIO IPC thread: " << this);
    ioThread_.join();

    YIO_LOG_INFO("ASIO IPC is stopped: " << this);
}

std::shared_ptr<ICallbackQueue> AsioAsyncWorker::workerQueue(std::string name) const {
    Y_VERIFY(asioCallbackPool_);
    Y_VERIFY(!name.empty());
    return asioCallbackPool_->createAsioCallbackQueue(name);
}

std::vector<std::weak_ptr<AsioAsyncObject>> AsioAsyncWorker::asyncObjects() {
    return asioAsyncObjectStorage_ ? asioAsyncObjectStorage_->asyncObjects() : std::vector<std::weak_ptr<AsioAsyncObject>>{};
}
