#include "asio_callback_queue.h"
#include "asio_callback_pool.h"

#include <util/system/yassert.h>

using namespace quasar;
using namespace quasar::ipc::detail::asio_ipc;

AsioCallbackQueue::AsioCallbackQueue(std::string name, std::shared_ptr<IAsioCallbackController> controller)
    : name_(std::move(name))
    , controller_(std::move(controller))
{
    Y_VERIFY(!name_.empty());
    Y_VERIFY(controller_);
}

AsioCallbackQueue::~AsioCallbackQueue() {
    controller_->notifyDestroy();
}

std::atomic<std::thread::id>& AsioCallbackQueue::worker() {
    return worker_;
}

std::atomic<int32_t>& AsioCallbackQueue::iterations() {
    return iterations_;
}

AsioCallbackQueue::Callback AsioCallbackQueue::pop() noexcept {
    Callback result;

    std::lock_guard lock(mutex_);
    if (!queue_.empty()) {
        result = std::move(queue_.front());
        queue_.pop_front();
    }

    return result;
}

void AsioCallbackQueue::shutdown() {
    std::lock_guard lock(mutex_);
    shutdown_ = true;
}

void AsioCallbackQueue::destroy() {
    shutdown();
    wait(AwatingType::ALTRUIST);

    std::lock_guard lock(mutex_);
    stopped_ = true;
}

std::string AsioCallbackQueue::name() const {
    return name_;
}

void AsioCallbackQueue::add(std::function<void()> callback) {
    add(std::move(callback), Lifetime::immortal.tracker());
}

void AsioCallbackQueue::add(std::function<void()> callback, Lifetime::Tracker tracker) {
    std::lock_guard lock(mutex_);
    if (Y_UNLIKELY(shutdown_)) {
        return;
    }
    queue_.push_back(Callback{std::move(callback), std::move(tracker)});
    controller_->notifyTask(weak_from_this());
}

bool AsioCallbackQueue::tryAdd(std::function<void()> callback) {
    add(std::move(callback));
    return true;
}

bool AsioCallbackQueue::tryAdd(std::function<void()> callback, Lifetime::Tracker tracker) {
    add(std::move(callback), std::move(tracker));
    return true;
}

void AsioCallbackQueue::addDelayed(std::function<void()> callback, std::chrono::milliseconds timeOut) {
    std::lock_guard lock(mutex_);
    if (Y_UNLIKELY(shutdown_)) {
        return;
    }
    controller_->scheduleTask(weak_from_this(), std::move(callback), timeOut, Lifetime::immortal.tracker());
}

void AsioCallbackQueue::addDelayed(std::function<void()> callback, std::chrono::milliseconds timeOut, Lifetime::Tracker tracker) {
    std::lock_guard lock(mutex_);
    if (Y_UNLIKELY(shutdown_)) {
        return;
    }
    controller_->scheduleTask(weak_from_this(), std::move(callback), timeOut, std::move(tracker));
}

void AsioCallbackQueue::wait(AwatingType awatingType) {
    if (worker_.load() == std::this_thread::get_id()) {
        throw std::logic_error("DEAD LOCK in wait() call: can't awaiting inside AsioCallbackQueue");
    }

    std::condition_variable cv;
    bool repeat = false;
    do {
        auto ready = false;
        auto marker = [&] {
            std::lock_guard lock(mutex_);
            ready = true;
            cv.notify_all();
        };
        std::unique_lock lock(mutex_);
        if (Y_UNLIKELY(stopped_)) {
            return;
        }
        queue_.push_back(Callback{std::move(marker), Lifetime::immortal.tracker()});
        controller_->notifyTask(weak_from_this());
        cv.wait(lock, [&] { return stopped_ || ready; });
        repeat = !stopped_ && awatingType != AwatingType::EGOIST && queue_.size();
    } while (repeat);

    if (awatingType == AwatingType::ALTRUIST) {
        // Because it is impossible to synchronously check "queue_.size()", there may
        // be a situation when the last task is still running, and we have already exited
        // the waiting loop. To do this, call "wait" for the grant again.
        wait(AwatingType::EGOIST);
    }
}

bool AsioCallbackQueue::isWorkingThread() const noexcept {
    return worker_.load() == std::this_thread::get_id();
}

size_t AsioCallbackQueue::size() const {
    std::lock_guard lock(mutex_);
    return queue_.size();
}

void AsioCallbackQueue::setMaxSize(size_t /*size*/) {
}
