#include "asio_channel.h"

#include "asio_logging.h"

#include <yandex_io/libs/logging/logging.h>

using namespace quasar;
using namespace quasar::ipc::detail::asio_ipc;

namespace {
    constexpr size_t BUFFER_SIZE = 128;

    // local no service name: `1234`
    // remote no service name: `11.22.33.44:1234`
    // local: `<serviceName>:1234`
    // remote: `<serviceName>:11.22.33.44:1234`
    std::string makeServiceDescription(const std::string& serviceName, const asio::ip::tcp::socket& sock) {
        std::ostringstream out;
        out << serviceName << ':';

        asio::error_code ec;
        if (auto epLocal = sock.local_endpoint(ec); !ec) {
            out << epLocal.port() << ':';
        }

        if (auto epRemote = sock.remote_endpoint(ec); !ec) {
            if (!epRemote.address().is_loopback()) {
                out << epRemote.address() << ':';
            }
            out << epRemote.port();
        }

        return out.str();
    }

} // namespace

AsioChannel::AsioChannel(std::shared_ptr<AsioAsyncWorker> worker, asio::ip::tcp::socket sock, const std::string& serviceName)
    : AsioAsyncObject(std::move(worker))
    , serviceDescription_(makeServiceDescription(serviceName, sock))
    , strand_(ioContext().get_executor())
    , socket_(std::move(sock))
    , readBuffer_(BUFFER_SIZE)
{
}

AsioChannel::~AsioChannel() {
    YIO_LOG_DEBUG(*this << ": destroy");
}

void AsioChannel::debugPrintDescription(std::ostream& out) const {
    out << "<Channel";
    if (const auto& service = serviceDescription(); !service.empty()) {
        out << ':' << service;
    }
    out << ' ' << this << '>';
}

void AsioChannel::doAsyncStart() {
    YIO_LOG_TRACE(*this << ": start");
    asio::dispatch(strand_, [weakThis = weak_from_this()] {
        if (auto this_ = weakThis.lock()) {
            YIO_LOG_DEBUG(*this_ << ": async start");
            this_->onIpcConnect();
            this_->asyncReadNext();
        }
    });
}

void AsioChannel::doAsyncShutdown() {
    YIO_LOG_TRACE(*this << ": shutdown");
    asio::dispatch(strand_, [weakThis = weak_from_this()] {
        if (auto this_ = weakThis.lock()) {
            YIO_LOG_DEBUG(*this_ << ": async shutdown");
            if (this_->socket_.is_open()) {
                this_->closeSocket();
            }
        }
    });
}

void AsioChannel::closeSocket() {
    // TODO: call onIpcConnectionError
    // TODO: synchronize / execute in io_context?
    // FIXME: close() really races with other socket ops

    if (socket_.is_open()) {
        asio::error_code ec;
        socket_.close(ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": close");

        onIpcDisconnect();
    }
}

void AsioChannel::asyncReadNext() {
    YIO_LOG_TRACE(*this << ": read next (buffer size: " << readBuffer_.size() << ")");
    auto buffer = asio::mutable_buffer(readBuffer_.data(), readBuffer_.size());
    socket_.async_read_some(buffer, asio::bind_executor(strand_, [weakThis = weak_from_this()](const auto& ec, auto bytes) {
                                if (auto this_ = weakThis.lock()) {
                                    this_->onReadData(ec, bytes);
                                }
                            }));
}

void AsioChannel::onReadData(const asio::error_code& ec, size_t bytesTransferred) {
    ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": async read data: " << bytesTransferred << " bytes");
    bool isOk = false;
    if (!ec) {
        auto buffer = asio::const_buffer(readBuffer_.data(), bytesTransferred);
        isOk = readTokenizer_.pushData(buffer, [this](auto messageBuffer) {
            return onIpcMessage(messageBuffer);
        });
    }

    if (isOk) {
        if (readTokenizer_.maxSize() > readBuffer_.size()) {
            // TODO: a better adaptive policy or a proper allocator?
            YIO_LOG_DEBUG(*this << ": adaptive resize buffer: " << readBuffer_.size() << " -> " << readTokenizer_.maxSize());
            readBuffer_.resize(readTokenizer_.maxSize());
        }
        asyncReadNext();
    } else {
        // TODO: read failed: break connection
        closeSocket();
    }
}

void AsioChannel::asyncSend(std::shared_ptr<TString> data) {
    if (data->size() == 0) {
        YIO_LOG_WARN(*this << ": empty send command, skipping");
        return;
    }

    YIO_LOG_TRACE(*this << ": send data: " << data->size() << " bytes");
    asio::dispatch(strand_, [weakThis = weak_from_this(), data = std::move(data)]() mutable {
        if (auto this_ = weakThis.lock()) {
            this_->asyncSendImpl(std::move(data));
        }
    });
}

void AsioChannel::asyncSendImpl(std::shared_ptr<TString> data) {
    std::unique_lock lock(sendQueueMutex_);
    const bool canSendImmediately = sendQueue_.empty();

    // TODO: check queue size to avoid overflow?
    sendQueue_.emplace_back(std::move(data));

    // Iff the send queue is non-empty, there is an ongoing send action sequence
    // via repeated asyncSendNextLocked() calls
    if (canSendImmediately) {
        YIO_LOG_TRACE(*this << ": immediate send");
        // Initiate a new send action sequence
        asyncSendNextLocked(lock);
    } else {
        YIO_LOG_TRACE(*this << ": queued send");
        // An ongoing send action sequence will pick up this buffer
    }
}

void AsioChannel::asyncSendNextLocked(std::unique_lock<std::mutex>& lock) {
    if (sendQueue_.empty()) {
        YIO_LOG_TRACE(*this << ": async send next: empty queue");
        return;
    }

    YIO_LOG_TRACE(*this << ": async send next: " << sendQueue_.size() << " queued buffers");
    auto buffers = prepareSendBufferLocked();
    lock.unlock();

    YIO_LOG_TRACE(*this << ": async send next: prepared " << buffers.size() << " buffers; total size = " << asio::buffer_size(buffers));
    socket_.async_write_some(buffers, asio::bind_executor(strand_, [weakThis = weak_from_this()](const auto& ec, auto bytesTransferred) {
                                 if (auto this_ = weakThis.lock()) {
                                     this_->onSendData(ec, bytesTransferred);
                                 }
                             }));
}

std::vector<asio::const_buffer> AsioChannel::prepareSendBufferLocked() const {
    std::vector<asio::const_buffer> buffers;
    buffers.reserve(sendQueue_.size());
    size_t offset = sendOffset_;
    for (const auto& item : sendQueue_) {
        auto buffer = asio::const_buffer(item->data(), item->size());
        if (offset > 0) {
            buffer += offset;
            offset = 0;
        }
        buffers.push_back(buffer);
    }
    return buffers;
}

void AsioChannel::onSendData(const asio::error_code& ec, size_t bytesTransferred) {
    (void)bytesTransferred;

    ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": async send data: " << bytesTransferred << " bytes");
    if (!ec) {
        std::unique_lock lock(sendQueueMutex_);
        sendOffset_ += bytesTransferred;
        while (!sendQueue_.empty() && sendQueue_.front()->size() <= sendOffset_) {
            sendOffset_ -= sendQueue_.front()->size();
            sendQueue_.pop_front();
        }

        asyncSendNextLocked(lock);
    } else {
        // TODO: write failed: break connection
        closeSocket();
    }
}
