#pragma once

#include "asio_async_worker.h"
#include "asio_tcp_connector.h"

#include <yandex_io/libs/ipc/i_connector.h>

#include <memory>
#include <string>
#include <unordered_map>

#include <asio.hpp>

namespace quasar::ipc::detail::asio_ipc {

    class AsioConnector: public AsioAsyncObject, public std::enable_shared_from_this<AsioConnector> {
    public:
        using OnConnect = ipc::IConnector::OnConnect;
        using OnDisconnect = ipc::IConnector::OnDisconnect;
        using OnConnectionError = ipc::IConnector::OnConnectionError;
        using OnMessage = ipc::IConnector::MessageHandler;
        using OnDone = ipc::IConnector::OnDone;
        using OnError = ipc::IConnector::OnError;

        struct Callbacks {
            Callbacks(std::shared_ptr<ICallbackQueue> callbackQueue);

            std::shared_ptr<ICallbackQueue> queue;

            OnConnect onConnect;
            OnDisconnect onDisconnect;
            OnConnectionError onConnectionError;
            OnMessage onMessage;
        };

        class RequestHandlerMap: public AsioAsyncObject, public std::enable_shared_from_this<RequestHandlerMap> {
        public:
            RequestHandlerMap(std::shared_ptr<AsioAsyncWorker> worker);
            ~RequestHandlerMap();

            void debugPrintDescription(std::ostream& out) const override;

            std::string add(OnDone onDone, OnError onError, std::chrono::milliseconds timeout);
            bool notifyDone(const std::string& key, const SharedMessage& message);
            bool notifyError(const std::string& key, const std::string& errorMessage);

        private:
            void doAsyncStart() override;
            void doAsyncShutdown() override;

        private:
            struct Entry {
                asio::steady_timer timer;
                OnDone onDone;
                OnError onError;

                Entry(asio::io_context& context, OnDone onDone_, OnError onError_)
                    : timer(context)
                    , onDone(std::move(onDone_))
                    , onError(std::move(onError_))
                {
                }
            };

        private:
            asio::strand<asio::io_context::executor_type> strand_;

            mutable std::mutex handlersMutex_;
            std::unordered_map<std::string, Entry> handlers_;
        };

        class ConnectorChannel: public AsioChannel {
        public:
            using OnDisconnect = std::function<void(ConnectorChannel*)>;

        public:
            ConnectorChannel(std::shared_ptr<AsioAsyncWorker> asyncWorker, std::shared_ptr<AsioConnector::Callbacks> callbacks, asio::ip::tcp::socket sock, const std::string& serviceName, OnDisconnect onDisconnect);

            void debugPrintDescription(std::ostream& out) const override;

            void sendMessageImpl(const Message& message);
            void sendRequestImpl(Message& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout);

        protected:
            void onIpcConnect() override;
            void onIpcDisconnect() override;
            void onIpcConnectionError(std::string errorMessage) override;
            bool onIpcMessage(asio::const_buffer buffer) override;

        private:
            void handleIncomingPayload(std::string_view payload);
            bool handleRequestResponse(const SharedMessage& message);

        private:
            std::shared_ptr<AsioConnector::Callbacks> callbacks_;
            OnDisconnect onDisconnect_;

            mutable std::mutex requestHandlersMutex_;
            std::shared_ptr<RequestHandlerMap> requestHandlers_;
        };

    public:
        AsioConnector(std::string serviceName, AsioTcpConnector::Address address, std::shared_ptr<AsioAsyncWorker> worker, std::shared_ptr<Callbacks> callbacks);

        void debugPrintDescription(std::ostream& out) const override;

        bool sendMessageImpl(const Message& message);

        std::shared_ptr<ConnectorChannel> lockChannel();

        bool isConnected() const;

        bool waitUntilConnected(const std::chrono::milliseconds& timeout) const;
        bool waitUntilDisconnected(const std::chrono::milliseconds& timeout) const;

    private:
        void doAsyncStart() override;
        void doAsyncShutdown() override;
        void asyncConnect();
        std::shared_ptr<AsioTcpConnector> createConnector();
        void setTcpChannel(asio::ip::tcp::socket peer);
        void setChannel(std::shared_ptr<ConnectorChannel> channel);

    private:
        std::string serviceName_;
        AsioTcpConnector::Address address_;
        std::shared_ptr<Callbacks> callbacks_;

        mutable std::mutex connectorMutex_;
        std::shared_ptr<AsioTcpConnector> connector_;

        mutable std::mutex channelMutex_;
        mutable SteadyConditionVariable channelCV_;
        std::shared_ptr<ConnectorChannel> channel_;
    };

} // namespace quasar::ipc::detail::asio_ipc
