#include "asio_connector_facade.h"

#include "asio_logging.h"

#include <yandex_io/libs/ipc/helpers.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <util/system/yassert.h>

using namespace quasar::ipc;
using namespace quasar::ipc::detail::asio_ipc;
using namespace std::chrono_literals;

AsioConnectorFacade::AsioConnectorFacade(std::string serviceName,
                                         std::shared_ptr<YandexIO::Configuration> configuration,
                                         std::shared_ptr<AsioAsyncWorker> worker)
    : configuration_(std::move(configuration))
    , serviceName_(std::move(ipc::helpers::checkServiceName(serviceName)))
    , worker_(std::move(worker))
    , callbacks_(std::make_shared<AsioConnector::Callbacks>(worker_->workerQueue(makeString("con:", serviceName_, ":", this))))
{
    Y_VERIFY(configuration_ != nullptr);
}

AsioConnectorFacade::~AsioConnectorFacade() {
    shutdown();
}

void AsioConnectorFacade::shutdown() {
    auto lock = std::scoped_lock{mutex_};
    if (impl_) {
        doSyncShutdownUnlocked();
    }
}

void AsioConnectorFacade::doSyncShutdownUnlocked() {
    YIO_LOG_DEBUG(*this << ": initiate shutdown");
    impl_->asyncShutdown();
    // The connector should stop trying to reconnect soon

    // Synchronously finalize all activity
    YIO_LOG_DEBUG(*this << ": wait for the channel to die");
    while (!impl_->waitUntilDisconnected(1s)) {
    }

    YIO_LOG_DEBUG(*this << ": wait for the connector impl to die");
    impl_ = nullptr;
    implDeath_.get();

    YIO_LOG_DEBUG(*this << ": shutdown complete");
}

const std::string& AsioConnectorFacade::serviceName() const {
    return serviceName_;
}

void AsioConnectorFacade::setConnectHandler(OnConnect handler) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ == nullptr);

    callbacks_->onConnect = std::move(handler);
}

void AsioConnectorFacade::setDisconnectHandler(OnDisconnect handler) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ == nullptr);

    callbacks_->onDisconnect = std::move(handler);
}

void AsioConnectorFacade::setConnectionErrorHandler(OnConnectionError handler) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ == nullptr);

    callbacks_->onConnectionError = std::move(handler);
}

void AsioConnectorFacade::setMessageHandler(MessageHandler handler) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ == nullptr);

    callbacks_->onMessage = std::move(handler);
}

void AsioConnectorFacade::setSilentMode(bool silentMode) {
    // TODO
    (void)silentMode;
}

bool AsioConnectorFacade::sendMessage(const SharedMessage& message) {
    return doSendMessage(*message);
}

bool AsioConnectorFacade::sendMessage(Message&& message) {
    return doSendMessage(message);
}

bool AsioConnectorFacade::doSendMessage(const Message& message) {
    auto lock = std::scoped_lock{mutex_};
    if (!impl_) {
        YIO_LOG_WARN(*this << ": cannot send message: not configured")
        return false;
    }

    if (auto channel = impl_->lockChannel()) {
        channel->sendMessageImpl(message);
        return true;
    }

    YIO_LOG_WARN(*this << ": cannot send message: not connected")
    return false;
}

void AsioConnectorFacade::sendRequest(UniqueMessage&& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout) {
    doSendRequest(*message, std::move(onDone), std::move(onError), timeout);
}

void AsioConnectorFacade::sendRequest(Message&& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout) {
    doSendRequest(message, std::move(onDone), std::move(onError), timeout);
}

void AsioConnectorFacade::doSendRequest(Message& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout) {
    Y_VERIFY(onDone);
    Y_VERIFY(onError);

    auto lock = std::scoped_lock{mutex_};
    auto queue = callbacks_->queue;
    auto realOnError =
        [queue, onError = std::move(onError)](const std::string& errorMessage) mutable {
            // called at most once: safe to move-from onError
            queue->add([onError = std::move(onError), errorMessage] { onError(errorMessage); });
        };

    if (Y_LIKELY(impl_)) {
        auto channel = impl_->lockChannel();
        if (Y_LIKELY(channel)) {
            channel->sendRequestImpl(message, std::move(onDone), std::move(realOnError), timeout);
        } else {
            realOnError("cannot send message: not connected");
        }
    } else {
        realOnError("cannot send message: not configured");
    }
}

SharedMessage AsioConnectorFacade::sendRequestSync(UniqueMessage&& message, std::chrono::milliseconds timeout) {
    return doSendRequestSync(*message, timeout);
}

SharedMessage AsioConnectorFacade::sendRequestSync(Message&& message, std::chrono::milliseconds timeout) {
    return doSendRequestSync(message, timeout);
}

SharedMessage AsioConnectorFacade::doSendRequestSync(Message& message, std::chrono::milliseconds timeout) {
    auto lock = std::scoped_lock{mutex_};

    std::promise<SharedMessage> requestDone;
    auto onDone = [&](const SharedMessage& response) {
        requestDone.set_value(response);
    };

    auto onError = [&](const std::string& errorMessage) {
        requestDone.set_exception(std::make_exception_ptr(
            std::runtime_error("Cannot send request: " + errorMessage)));
    };

    if (Y_LIKELY(impl_)) {
        auto channel = impl_->lockChannel();
        if (Y_LIKELY(channel)) {
            channel->sendRequestImpl(message, std::move(onDone), std::move(onError), timeout);
        } else {
            onError("cannot send message: not connected");
        }
    } else {
        onError("cannot send message: not configured");
    }

    return requestDone.get_future().get();
}

bool AsioConnectorFacade::tryConnectToService() {
    if (configuration_->getFullConfig().isMember(serviceName_)) {
        connectToService();
        return true;
    }

    YIO_LOG_DEBUG(*this << ": skip connect: there is no \"" << serviceName_ << "\" entry in config");
    return false;
}

void AsioConnectorFacade::connectToService() {
    auto port = configuration_->getServicePort(serviceName_);

    connectToTcpHost("localhost", port);
}

void AsioConnectorFacade::connectToTcpLocal(int port) {
    connectToTcpHost("localhost", port);
}

void AsioConnectorFacade::connectToTcpHost(const std::string& hostname, int port) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ == nullptr);

    auto address = AsioTcpConnector::Address{hostname, port};
    impl_ = worker_->createWithDeathFuture<AsioConnector>(implDeath_, serviceName_, std::move(address), worker_, callbacks_);
    YIO_LOG_DEBUG(*this << ": create connector impl " << *impl_);
    impl_->asyncStart();
}

bool AsioConnectorFacade::isConnected() const {
    auto lock = std::scoped_lock{mutex_};

    return impl_ != nullptr && impl_->isConnected();
}

void AsioConnectorFacade::waitUntilConnected() const {
    while (!waitUntilConnected(1s)) {
        // try again indefinitely
    }
}

void AsioConnectorFacade::waitUntilDisconnected() const {
    while (!waitUntilDisconnected(1s)) {
        // try again indefinitely
    }
}

bool AsioConnectorFacade::waitUntilConnected(const std::chrono::milliseconds& timeout) const {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ != nullptr);

    return impl_->waitUntilConnected(timeout);
}

bool AsioConnectorFacade::waitUntilDisconnected(const std::chrono::milliseconds& timeout) const {
    auto lock = std::scoped_lock{mutex_};

    if (impl_) {
        return impl_->waitUntilDisconnected(timeout);
    } else {
        return true;
    }
}

std::ostream& quasar::ipc::detail::asio_ipc::operator<<(std::ostream& out, const AsioConnectorFacade& connector) {
    out << "<Connector.Facade";
    out << ':' << connector.serviceName_;
    out << ' ' << &connector << '>';
    return out;
}
