#pragma once

#include "asio_async_worker.h"
#include "asio_tcp_acceptor.h"

#include <yandex_io/libs/ipc/i_server.h>

namespace quasar::ipc::detail::asio_ipc {

    class AsioServer: public AsioAsyncObject, public std::enable_shared_from_this<AsioServer> {
    public:
        using ClientHandler = ipc::IServer::ClientHandler;
        using MessageHandler = ipc::IServer::MessageHandler;

        struct Callbacks {
            Callbacks(std::shared_ptr<ICallbackQueue> callbackQueue);

            std::shared_ptr<ICallbackQueue> queue;

            ClientHandler onClientConnected;
            ClientHandler onClientDisconnected;
            MessageHandler onMessage;
        };

        class ServerChannel: public AsioChannel {
        public:
            using OnDisconnect = std::function<void(ServerChannel*)>;

        public:
            ServerChannel(std::shared_ptr<AsioAsyncWorker> asyncWorker, std::shared_ptr<AsioServer::Callbacks> callbacks, asio::ip::tcp::socket sock, const std::string& serviceName, OnDisconnect onDisconnect);

            void debugPrintDescription(std::ostream& out) const override;

            std::shared_ptr<IServer::IClientConnection> sharedClientConnection();

        protected:
            void onIpcConnect() override;
            void onIpcDisconnect() override;
            void onIpcConnectionError(std::string errorMessage [[maybe_unused]]) override;
            bool onIpcMessage(asio::const_buffer buffer) override;

        private:
            void handleIncomingPayload(std::string_view payload);

        private:
            std::shared_ptr<AsioServer::Callbacks> callbacks_;
            OnDisconnect onDisconnect_;

            std::mutex sharedClientConnectionMutex_;
            std::shared_ptr<IServer::IClientConnection> sharedClientConnection_;
        };

    public:
        AsioServer(std::string serviceName, AsioTcpAcceptor::Address address, std::shared_ptr<AsioAsyncWorker> worker, std::shared_ptr<Callbacks> callbacks);

        void debugPrintDescription(std::ostream& out) const override;

        int getConnectedClientCount() const;
        int port() const;

        bool waitConnectionsAtLeast(size_t count, std::chrono::milliseconds timeout) const;
        bool waitConnectionsAtMost(size_t count, std::chrono::milliseconds timeout) const;
        bool waitListening(bool targetStatus, std::chrono::milliseconds timeout) const;
        void sendToAll(const Message& message);

    private:
        class WeakClientConnection;

    private:
        void notifyListen();
        void notifyStopListening();
        void addTcpChannel(asio::ip::tcp::socket peer);
        void addChannel(std::shared_ptr<AsioChannel> channel);
        void removeChannel(std::shared_ptr<AsioChannel> channel);

    private:
        void doAsyncStart() override;
        void doAsyncShutdown() override;
        void doStartAcceptor();
        std::shared_ptr<AsioTcpAcceptor> createAcceptor();

    private:
        std::string serviceName_;
        AsioTcpAcceptor::Address address_;
        std::shared_ptr<Callbacks> callbacks_;

        mutable std::mutex acceptorMutex_;
        mutable SteadyConditionVariable acceptorCV_;
        std::shared_ptr<AsioTcpAcceptor> acceptor_;
        bool isListening_{false};

        mutable std::mutex channelsMutex_;
        mutable SteadyConditionVariable channelsCV_;
        std::set<std::shared_ptr<AsioChannel>> channels_;
    };

} // namespace quasar::ipc::detail::asio_ipc
