#include "asio_server_facade.h"

#include "asio_logging.h"
#include "serialize.h"

#include <yandex_io/libs/ipc/helpers.h>

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>

#include <yandex_io/protos/quasar_proto.pb.h>

#include <util/system/yassert.h>

YIO_DEFINE_LOG_MODULE("asio_ipc");

using namespace quasar::ipc::detail::asio_ipc;
using namespace std::chrono_literals;

AsioServerFacade::AsioServerFacade(std::string serviceName,
                                   std::shared_ptr<YandexIO::Configuration> configuration,
                                   std::shared_ptr<AsioAsyncWorker> worker)
    : configuration_(std::move(configuration))
    , serviceName_(std::move(ipc::helpers::checkServiceName(serviceName)))
    , worker_(std::move(worker))
    , callbacks_(std::make_shared<AsioServer::Callbacks>(worker_->workerQueue(makeString("srv:", serviceName_, ":", this))))
{
    Y_VERIFY(configuration_ != nullptr);

    YIO_LOG_INFO(*this << ": create server");
}

AsioServerFacade::~AsioServerFacade() {
    YIO_LOG_INFO(*this << ": destroy");

    shutdown();
}

void AsioServerFacade::doListenTcpLocal(int port) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ == nullptr);

    auto address = AsioTcpAcceptor::Address{port};
    impl_ = worker_->createWithDeathFuture<AsioServer>(implDeath_, serviceName_, address, worker_, callbacks_);
    YIO_LOG_DEBUG(*this << ": create server impl " << *impl_);
    impl_->asyncStart();
}

void AsioServerFacade::shutdown() {
    auto lock = std::scoped_lock{mutex_};
    if (impl_) {
        doSyncShutdownUnlocked();
    }
}

void AsioServerFacade::doSyncShutdownUnlocked() {
    YIO_LOG_DEBUG(*this << ": initiate shutdown");
    impl_->asyncShutdown();

    // Synchronously finalize all activity
    YIO_LOG_DEBUG(*this << ": wait for the channel to die");
    while (!impl_->waitConnectionsAtMost(0, 1s)) {
    }

    YIO_LOG_DEBUG(*this << ": wait for the server impl to die");
    impl_ = nullptr;
    implDeath_.get();

    YIO_LOG_DEBUG(*this << ": shutdown complete");
}

void AsioServerFacade::setClientConnectedHandler(ClientHandler handler) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ == nullptr);

    callbacks_->onClientConnected = std::move(handler);
}

void AsioServerFacade::setClientDisconnectedHandler(ClientHandler handler) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ == nullptr);

    callbacks_->onClientDisconnected = std::move(handler);
}

const std::string& AsioServerFacade::serviceName() const {
    return serviceName_;
}

void AsioServerFacade::setMessageHandler(MessageHandler handler) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ == nullptr);

    callbacks_->onMessage = std::move(handler);
}

void AsioServerFacade::listenService() {
    try {
        auto serviceConfig = configuration_->getServiceConfig(serviceName_);
        const int port = tryGetInt(serviceConfig, "port", -1);
        // do not allow to start server without proper port
        Y_VERIFY(port != -1);
        // TODO: load "bind" from config?

        doListenTcpLocal(port);
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("AsioServerFacade.ListenService.Exception", *this << ": failed to start server for \"" << serviceName_ << "\": " << ex.what());
        throw;
    } catch (...) {
        YIO_LOG_ERROR_EVENT("AsioServerFacade.ListenService.UnknownError", *this << ": failed to start server for \"" << serviceName_ << "\": unexpected exception");
        throw;
    }
}

void AsioServerFacade::listenTcpHost(const std::string& /* hostname */, int port) {
    doListenTcpLocal(port); // FIXME: pass proper hostname
}

int AsioServerFacade::listenTcpLocal(int port) {
    doListenTcpLocal(port);
    return port;
}

int AsioServerFacade::port() const {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ != nullptr);

    return impl_->port();
}

int AsioServerFacade::getConnectedClientCount() const {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ != nullptr);

    return impl_->getConnectedClientCount();
}

void AsioServerFacade::waitConnectionsAtLeast(size_t count) {
    while (!waitConnectionsAtLeast(count, 1s)) {
        // try again indefinitely
    }
}

bool AsioServerFacade::waitConnectionsAtLeast(size_t count, std::chrono::milliseconds timeout) {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ != nullptr);

    return impl_->waitConnectionsAtLeast(count, timeout);
}

void AsioServerFacade::waitListening() const {
    while (!waitListening(1s)) {
        // try again indefinitely
    }
}

bool AsioServerFacade::waitListening(std::chrono::milliseconds timeout) const {
    auto lock = std::scoped_lock{mutex_};
    Y_VERIFY(impl_ != nullptr);

    return impl_->waitListening(true, timeout);
}

void AsioServerFacade::sendToAll(const SharedMessage& message) {
    doSendToAll(*message);
}

void AsioServerFacade::sendToAll(Message&& message) {
    doSendToAll(message);
}

void AsioServerFacade::doSendToAll(const Message& message) {
    auto lock = std::scoped_lock{mutex_};
    if (!impl_) {
        YIO_LOG_WARN(*this << ": cannot broadcast message: not configured")
        return;
    }

    impl_->sendToAll(message);
}

std::ostream& quasar::ipc::detail::asio_ipc::operator<<(std::ostream& out, const AsioServerFacade& server) {
    out << "<Server.Facade";
    out << ':' << server.serviceName_;
    out << ' ' << &server << '>';
    return out;
}
