#include "asio_tcp_acceptor.h"

#include "asio_logging.h"

#include <yandex_io/libs/logging/logging.h>

#include <util/network/socket.h>

using namespace quasar::ipc::detail::asio_ipc;
using namespace std::chrono_literals;

void AsioTcpAcceptor::Callbacks::verify() const {
    Y_VERIFY(onListen != nullptr);
    Y_VERIFY(onStopListening != nullptr);
    Y_VERIFY(onConnect != nullptr);
}

AsioTcpAcceptor::AsioTcpAcceptor(std::shared_ptr<AsioAsyncWorker> worker, std::string serviceName, Address address, Callbacks callbacks)
    : AsioAsyncObject(std::move(worker))
    , serviceName_(std::move(serviceName))
    , address_(address)
    , callbacks_(std::move(callbacks))
    , strand_(ioContext().get_executor())
    , acceptor_(ioContext())
    , retryTimer_(ioContext())
{
    callbacks_.verify();

    YIO_LOG_TRACE(*this << ": create"); // the parent should print a DEBUG message
}

AsioTcpAcceptor::~AsioTcpAcceptor() {
    YIO_LOG_DEBUG(*this << ": destroy");
}

void AsioTcpAcceptor::debugPrintDescription(std::ostream& out) const {
    out << "<TcpAcceptor";
    out << ':' << serviceName_;
    out << ':' << address_.port;
    out << ' ' << this << '>';
}

void AsioTcpAcceptor::doAsyncStart() {
    YIO_LOG_TRACE(*this << ": start");
    asio::dispatch(strand_, [weakThis = weak_from_this()] {
        if (auto this_ = weakThis.lock()) {
            this_->listen();
        }
    });
}

void AsioTcpAcceptor::doAsyncShutdown() {
    YIO_LOG_TRACE(*this << ": shutdown");
    asio::dispatch(strand_, [weakThis = weak_from_this()] {
        if (auto this_ = weakThis.lock()) {
            YIO_LOG_TRACE(*this_ << ": async shutdown");
            asio::error_code ec;

            this_->retryTimer_.cancel(ec);
            ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this_ << ": cancel timer");

            this_->acceptor_.close(ec);
            ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this_ << ": close acceptor socket");
        }
    });
}

void AsioTcpAcceptor::listen() {
    YIO_LOG_TRACE(*this << ": listen");

    auto setupAcceptor = [this](auto& acceptor, const auto& endpoint) -> asio::error_code {
        asio::error_code ec;

        acceptor.open(endpoint.protocol(), ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": acceptor open");
        if (ec) {
            return ec;
        }

        try {
            SetCloseOnExec(acceptor.native_handle(), true);
        } catch (const TSystemError& err) {
            YIO_LOG_WARN(*this << ": acceptor close on exec: " << err.what());
        }

        acceptor.non_blocking(true, ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": acceptor non_blocking");
        if (ec) {
            return ec;
        }

        acceptor.set_option(asio::socket_base::reuse_address(true), ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": acceptor set_option(reuse_address): ");
        if (ec) {
            return ec;
        }

        acceptor.bind(endpoint, ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": acceptor bind endpoint '" << endpoint << "'");
        if (ec) {
            return ec;
        }

        acceptor.listen(asio::socket_base::max_listen_connections, ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": acceptor listen");
        return ec;
    };

    auto acceptor = asio::ip::tcp::acceptor(ioContext());
    auto endpoint = asio::ip::tcp::endpoint(asio::ip::address_v4::loopback(), address_.port);

    if (auto ec = setupAcceptor(acceptor, endpoint); !ec) {
        // success
        acceptor_ = std::move(acceptor);
        callbacks_.onListen();
        asyncAccept();
    } else {
        asyncRetryListen(ec);
    }
}

void AsioTcpAcceptor::asyncRetryListen(const asio::error_code& ec) {
    if (ec == asio::error::operation_aborted) {
        YIO_LOG_TRACE(*this << ": will not retry, operation canceled");
        callbacks_.onStopListening();
        return;
    }

    if (ec == asio::error::interrupted) {
        YIO_LOG_TRACE(*this << ": interrupted by a syscall, retry immediately");
        listen();
        return;
    }

    YIO_LOG_TRACE(*this << ": retry listen");
    // FIXME: calc timeout by retry policy
    retryTimer_.expires_after(ec ? 1s : 0s);
    retryTimer_.async_wait(asio::bind_executor(strand_, [weakThis = weak_from_this()](const auto& ec) {
        if (auto this_ = weakThis.lock()) {
            this_->onRetryListen(ec);
        }
    }));
}

void AsioTcpAcceptor::onRetryListen(const asio::error_code& ec) {
    ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": async retry");
    if (ec) {
        // FIXME: aborted
        // FIXME: interrupted call should preserve its deadline
        asyncRetryListen(ec);
    } else {
        listen();
    }
}

void AsioTcpAcceptor::asyncAccept() {
    YIO_LOG_TRACE(*this << ": accept");
    acceptor_.async_accept(asio::bind_executor(strand_, [weakThis = weak_from_this()](const auto& ec, auto peer) {
        if (auto this_ = weakThis.lock()) {
            this_->onAcceptIncomingConnection(ec, std::move(peer));
        }
    }));
}

void AsioTcpAcceptor::onAcceptIncomingConnection(const asio::error_code& acceptErrorCode, asio::ip::tcp::socket peer) {
    asio::error_code ec = acceptErrorCode;
    ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": async accept peer " << AsioTcpEndpointsLog{peer});

    try {
        SetCloseOnExec(peer.native_handle(), true);
        YIO_LOG_TRACE(*this << ": peer " << AsioTcpEndpointsLog{peer} << " close on exec: Ok");
    } catch (const TSystemError& err) {
        YIO_LOG_WARN(*this << ": peer " << AsioTcpEndpointsLog{peer} << " close on exec: " << err.what());
    }

    if (!ec) {
        peer.non_blocking(true, ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": peer " << AsioTcpEndpointsLog{peer} << " non_blocking");
    }

    if (!ec) {
        peer.set_option(asio::ip::tcp::no_delay(true), ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": peer " << AsioTcpEndpointsLog{peer} << " set_option(no_delay)");
    };

    if (!ec) {
        callbacks_.onConnect(std::move(peer));

        asyncAccept();
    } else {
        asyncRetryListen(ec);
    }
}
