#include "asio_tcp_connector.h"

#include "asio_logging.h"

#include <yandex_io/libs/logging/logging.h>

#include <util/generic/typetraits.h>
#include <util/network/socket.h>
#include <util/system/yassert.h>

using namespace quasar::ipc::detail::asio_ipc;
using namespace std::chrono_literals;

std::ostream& quasar::ipc::detail::asio_ipc::operator<<(std::ostream& out, const AsioTcpConnector::Address& address) {
    out << '[' << address.hostname << ':' << address.port << ']';
    return out;
}

void AsioTcpConnector::Callbacks::verify() const {
    Y_VERIFY(onConnect != nullptr);
    Y_VERIFY(onConnectionFailure != nullptr);
}

AsioTcpConnector::AsioTcpConnector(std::shared_ptr<AsioAsyncWorker> worker, std::string serviceName, Address address, Callbacks callbacks)
    : AsioAsyncObject(std::move(worker))
    , serviceName_(std::move(serviceName))
    , address_(std::move(address))
    , callbacks_(std::move(callbacks))
    , strand_(ioContext().get_executor())
    , resolver_(ioContext())
    , retryTimer_(ioContext())
{
    callbacks_.verify();

    YIO_LOG_TRACE(*this << ": create"); // the parent should print a DEBUG message
}

AsioTcpConnector::~AsioTcpConnector() {
    YIO_LOG_DEBUG(*this << ": destroy");
}

void AsioTcpConnector::debugPrintDescription(std::ostream& out) const {
    out << "<TcpConnector";
    out << ' ' << serviceName_;
    out << ' ' << address_;
    out << ' ' << this << '>';
}

void AsioTcpConnector::doAsyncStart() {
    YIO_LOG_TRACE(*this << ": start");
    asyncResolveEndpoint();
}

void AsioTcpConnector::doAsyncShutdown() {
    YIO_LOG_TRACE(*this << ": shutdown");
    asio::dispatch(strand_, [weakThis = weak_from_this()] {
        if (auto this_ = weakThis.lock()) {
            YIO_LOG_TRACE(*this_ << ": async shutdown");
            asio::error_code ec;
            this_->retryTimer_.cancel(ec);
            ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this_ << ": cancel timer");

            this_->resolver_.cancel();
        }
    });

    // TODO: cancel in-progress socket?
}

void AsioTcpConnector::asyncResolveEndpoint() {
    YIO_LOG_TRACE(*this << ": resolve endpoint " << address_);
    resolver_.async_resolve(address_.hostname, std::to_string(address_.port), asio::bind_executor(strand_, [weakThis = weak_from_this()](const auto& ec, auto results) {
                                if (auto this_ = weakThis.lock()) {
                                    this_->onResolveEndpoint(ec, std::move(results));
                                }
                            }));
}

void AsioTcpConnector::onResolveEndpoint(const asio::error_code& ec, asio::ip::tcp::resolver::results_type results) {
    ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": async resolve endpoint " << address_);
    if (ec) {
        // FIXME: retry
        asyncRetryConnect(ec);
    } else {
        if (YIO_LOG_TRACE_ENABLED()) {
            int idx = 0;
            for (const auto& result : results) {
                YIO_LOG_TRACE(*this << ": async resolved result " << idx << ": " << result.endpoint());
                ++idx;
            }
        }

        Y_VERIFY(results.begin() != results.end());
        asyncConnectEndpoint(results.begin(), results.end());
    }
}

void AsioTcpConnector::asyncConnectEndpoint(EndpointIterator epBegin, EndpointIterator epEnd) {
    const auto endpoint = epBegin->endpoint();
    YIO_LOG_TRACE(*this << ": connect endpoint " << endpoint);

    auto setupSocket = [this](auto& socket, const auto& endpoint) -> asio::error_code {
        asio::error_code ec;

        socket.open(endpoint.protocol(), ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": peer " << endpoint << " open");
        if (ec) {
            return ec;
        }

        try {
            SetCloseOnExec(socket.native_handle(), true);
        } catch (const TSystemError& err) {
            YIO_LOG_WARN(*this << ": peer " << endpoint << " close on exec: " << err.what());
        }

        socket.non_blocking(true, ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": peer " << endpoint << " non_blocking");
        if (ec) {
            return ec;
        }

        socket.set_option(asio::ip::tcp::no_delay(true), ec);
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": peer " << endpoint << " set_option(no_delay)");
        return ec;
    };

    auto sockPtr = std::make_shared<asio::ip::tcp::socket>(ioContext());
    if (auto ec = setupSocket(*sockPtr, endpoint); !ec) {
        // success
        sockPtr->async_connect(endpoint, asio::bind_executor(strand_, [weakThis = weak_from_this(), sockPtr, epBegin, epEnd](const auto& ec) {
                                   if (auto this_ = weakThis.lock()) {
                                       this_->onConnectEndpoint(ec, sockPtr, epBegin, epEnd);
                                   }
                               }));
    } else {
        asyncRetryConnect(ec);
    }
}

void AsioTcpConnector::onConnectEndpoint(const asio::error_code& ec, std::shared_ptr<asio::ip::tcp::socket> sockPtr, EndpointIterator epBegin, EndpointIterator epEnd) {
    auto epNext = epBegin;
    ++epNext;

    if (ec == asio::error::connection_refused) {
        YIO_LOG_DEBUG(*this << ": connection to " << epBegin->endpoint().address() << " refused");

        if (epNext != epEnd) {
            YIO_LOG_TRACE(*this << ": try next endpoint");
            asyncConnectEndpoint(epNext, epEnd);
            return;
        }
    } else {
        ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": async connect " << epBegin->endpoint().address());
    }

    if (ec) {
        asyncRetryConnect(ec);
    } else {
        callbacks_.onConnect(std::move(*sockPtr));
    }
}

void AsioTcpConnector::asyncRetryConnect(const asio::error_code& ec) {
    if (ec == asio::error::operation_aborted) {
        YIO_LOG_TRACE(*this << ": will not retry, operation canceled");
        return;
    }

    if (ec == asio::error::interrupted) {
        YIO_LOG_TRACE(*this << ": interrupted by a syscall, retry immediately");
        // XXX: This might be too long a detour: ideally should only restart the last syscall
        asyncResolveEndpoint();
        return;
    }

    const auto retryAction = callbacks_.onConnectionFailure();
    std::visit([this](auto&& action) {
        using T = std::decay_t<decltype(action)>;
        if constexpr (std::is_same_v<T, StopTrying>) {
            YIO_LOG_TRACE(*this << ": requested to stop retrying");
        } else if constexpr (std::is_same_v<T, RetryAfter>) {
            YIO_LOG_TRACE(*this << ": requested to retry after " << action.delay.count() << "ms");
            retryTimer_.expires_after(action.delay);
            retryTimer_.async_wait(asio::bind_executor(strand_, [weakThis = weak_from_this()](const auto& ec) {
                if (auto this_ = weakThis.lock()) {
                    this_->onRetryConnect(ec);
                }
            }));
        } else {
            static_assert(TDependentFalse<T>, "non-exhaustive retry actions!");
        }
    }, retryAction);
}

void AsioTcpConnector::onRetryConnect(const asio::error_code& ec) {
    ASIO_IPC_RESULT_LOG(TRACE, WARN, ec, *this << ": async retry");
    if (ec) {
        // FIXME: aborted
        // FIXME: interrupted call should preserve its deadline
        asyncRetryConnect(ec);
    } else {
        asyncResolveEndpoint();
    }
}
