#include "length_value_tokenizer.h"

#include <util/system/byteorder.h>

#include <functional>

using namespace quasar::ipc::detail::asio_ipc;

namespace {
    constexpr size_t LENGTH_SIZE = sizeof(int32_t);
    constexpr int32_t MAX_LENGTH = 10 * 1024 * 1024; // FIXME: tune this?
} // namespace

LengthValueTokenizer::LengthValueTokenizer() = default;

bool LengthValueTokenizer::pushData(asio::const_buffer data, OnMessage onMessage) {
    appendData(data);

    if (currentData_.size() > maxSize_) {
        maxSize_ = currentData_.size();
    }

    const auto* begin = currentData_.data();
    const auto* end = currentData_.end();
    const auto* ptr = begin;

    bool isOk = true;
    while (ptr + LENGTH_SIZE <= end) {
        int32_t signedLength = 0;
        memcpy(&signedLength, ptr, LENGTH_SIZE);
        signedLength = LittleToHost(signedLength);
        if (signedLength < 0 || signedLength > MAX_LENGTH) {
            // bad framing
            isOk = false;
            break;
        }

        size_t length = signedLength;
        if (LENGTH_SIZE + length <= static_cast<size_t>(end - ptr)) {
            auto buffer = asio::const_buffer(ptr + LENGTH_SIZE, length);
            if (onMessage(buffer)) {
                ptr += LENGTH_SIZE + length;
            } else {
                // bad encoding
                isOk = false;
                break;
            }
        } else {
            // wait for more data
            break;
        }
    }

    if (ptr > begin) {
        advanceData(ptr - begin);
    }

    return isOk;
}

void LengthValueTokenizer::appendData(asio::const_buffer data) {
    const auto* begin = static_cast<const std::byte*>(data.data());
    const auto* end = begin + data.size();
    currentData_.insert(currentData_.end(), begin, end);
}

void LengthValueTokenizer::advanceData(size_t bytesToSkip) {
    currentData_.erase(currentData_.begin(), currentData_.begin() + bytesToSkip);
}
