#include "connections_state_holder.h"

#include <yandex_io/libs/logging/logging.h>

YIO_DEFINE_LOG_MODULE("ipc");

using namespace quasar;

ConnectionsStateHolder::ConnectionsStateHolder(const std::list<std::string>& services) {
    for (const auto& service : services) {
        connectionsMap_[service] = false; /* not connected */
    }
}

void ConnectionsStateHolder::connect(const std::string& service) {
    std::scoped_lock guard(mutex_);
    checkHasService(service);
    connectionsMap_[service] = true;
}

void ConnectionsStateHolder::addService(const std::string& service) {
    std::scoped_lock guard(mutex_);
    connectionsMap_[service] = false; /* not connected */
}

void ConnectionsStateHolder::deleteService(const std::string& service) {
    std::scoped_lock guard(mutex_);
    if (connectionsMap_.count(service)) {
        connectionsMap_.erase(service);
    }
}

void ConnectionsStateHolder::disconnect(const std::string& service) {
    std::scoped_lock guard(mutex_);
    checkHasService(service);
    connectionsMap_[service] = false;
}

bool ConnectionsStateHolder::isConnected(const std::string& service) {
    std::scoped_lock guard(mutex_);
    checkHasService(service);
    return connectionsMap_[service];
}

bool ConnectionsStateHolder::allConnected() {
    std::scoped_lock guard(mutex_);
    for (const auto& service : connectionsMap_) {
        if (!service.second) {
            YIO_LOG_DEBUG("Service " << service.first << " is not connected.");
            /* at least one of the services is not connected */
            return false;
        }
    }
    return true;
}

void ConnectionsStateHolder::checkHasService(const std::string& service) {
    if (!connectionsMap_.count(service)) {
        /* All expected services should be passed in constructor */
        YIO_LOG_ERROR_EVENT("ConnectionsStateHolder.UnexpectedService", "Unexpected service: " << service);
    }
}
