#include "quasar_connector.h"

#include <yandex_io/libs/ipc/helpers.h>

#include <yandex_io/protos/quasar_proto.pb.h>

using namespace quasar;
using namespace quasar::ipc;
using namespace quasar::ipc::detail::datacratic;

QuasarConnector::QuasarConnector(std::string serviceName, std::shared_ptr<YandexIO::Configuration> configuration)
    : impl_(std::move(ipc::helpers::checkServiceName(serviceName)), std::move(configuration))
{
}

QuasarConnector::QuasarConnector(std::string serviceName, std::shared_ptr<YandexIO::Configuration> configuration, const Lifetime& lifetime, std::shared_ptr<ICallbackQueue> callbackQueue)
    : impl_(std::move(ipc::helpers::checkServiceName(serviceName)), lifetime, std::move(configuration), std::move(callbackQueue))
{
}

QuasarConnector::~QuasarConnector() = default;

const std::string& QuasarConnector::serviceName() const {
    return impl_.serviceName();
}

void QuasarConnector::setConnectHandler(OnConnect handler) {
    impl_.setConnectHandler(std::move(handler));
}

void QuasarConnector::setDisconnectHandler(OnDisconnect handler) {
    impl_.setDisconnectHandler(std::move(handler));
}

void QuasarConnector::setConnectionErrorHandler(OnConnectionError handler) {
    impl_.setConnectionErrorHandler(std::move(handler));
}

void QuasarConnector::setMessageHandler(MessageHandler handler) {
    impl_.setMessageHandler(std::move(handler));
}

void QuasarConnector::setSilentMode(bool silentMode) {
    impl_.setSilentMode(silentMode);
}

bool QuasarConnector::sendMessage(const SharedMessage& message)
{
    return impl_.sendMessage(message.ref());
}

bool QuasarConnector::sendMessage(Message&& message) {
    return impl_.sendMessage(message);
}

void QuasarConnector::sendRequest(UniqueMessage&& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout) {
    impl_.sendRequest(*message, std::move(onDone), std::move(onError), timeout);
}

void QuasarConnector::sendRequest(Message&& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout) {
    impl_.sendRequest(message, std::move(onDone), std::move(onError), timeout);
}

SharedMessage QuasarConnector::sendRequestSync(UniqueMessage&& message, std::chrono::milliseconds timeout) {
    return impl_.sendRequestSync(*message, timeout);
}

SharedMessage QuasarConnector::sendRequestSync(Message&& message, std::chrono::milliseconds timeout) {
    return impl_.sendRequestSync(message, timeout);
}

bool QuasarConnector::tryConnectToService() {
    return impl_.tryConnectToService();
}

void QuasarConnector::connectToService() {
    impl_.connectToService();
}

void QuasarConnector::connectToTcpHost(const std::string& hostname, int port) {
    impl_.connectToTcpHost(hostname, port);
}

void QuasarConnector::connectToTcpLocal(int port) {
    impl_.connectToTcpHost("localhost", port);
}

bool QuasarConnector::isConnected() const {
    return impl_.isConnected();
}

void QuasarConnector::waitUntilConnected() const {
    impl_.waitUntilConnected();
}

void QuasarConnector::waitUntilDisconnected() const {
    impl_.waitUntilDisconnected();
}

bool QuasarConnector::waitUntilConnected(const std::chrono::milliseconds& duration) const {
    return impl_.waitUntilConnected(duration);
}

bool QuasarConnector::waitUntilDisconnected(const std::chrono::milliseconds& duration) const {
    return impl_.waitUntilDisconnected(duration);
}

void QuasarConnector::shutdown() {
    impl_.shutdown();
}
