#include "quasar_connector_impl.h"

#include "length_value_tokenizer.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/callback_queue.h>

#include <yandex_io/protos/quasar_proto.pb.h>

#include <future>
#include <tuple>

#include <sys/prctl.h>

YIO_DEFINE_LOG_MODULE("datacratic_ipc");

using namespace quasar;
using namespace quasar::ipc;
using namespace quasar::ipc::detail::datacratic;
using namespace quasar::proto;

QuasarConnectorImpl::QuasarConnectorImpl(std::string serviceName, Lifetime::Tracker tracker, std::shared_ptr<YandexIO::Configuration> configuration, std::shared_ptr<ICallbackQueue> callbackQueue)
    : configuration_(std::move(configuration))
    , tracker_(std::move(tracker))
    , callbackQueue_(callbackQueue)
    , serviceName_(std::move(serviceName))
{
    setPollingMode(MIN_CPU_POLLING);
    getTokenizer = []() {
        return std::make_shared<LengthValueTokenizer>();
    };
    auto checkExpires = [=](int64_t /* timeouts */) mutable {
        std::unique_lock<std::mutex> lock(handlersMutex_);

        std::vector<OnError> errorHandlers;
        errorHandlers.reserve(64);
        auto onExpire = [&](const std::string& /* key */, std::pair<OnDone, OnError> value) mutable {
            errorHandlers.push_back(std::move(value.second));
        };
        handlers_.expire(onExpire);
        lock.unlock();
        for (auto& handler : errorHandlers) {
            invoke(std::move(handler), "Timeout");
        }
    };

    addPeriodic(0.05 /* 50ms */, checkExpires);
}

QuasarConnectorImpl::QuasarConnectorImpl(std::string serviceName, std::shared_ptr<YandexIO::Configuration> configuration)
    : QuasarConnectorImpl(std::move(serviceName), Lifetime::Tracker{}, std::move(configuration), nullptr)
{
}

QuasarConnectorImpl::QuasarConnectorImpl(std::string serviceName, const Lifetime& lifetime, std::shared_ptr<YandexIO::Configuration> configuration, std::shared_ptr<ICallbackQueue> callbackQueue)
    : QuasarConnectorImpl(std::move(serviceName), lifetime.tracker(), std::move(configuration), std::move(callbackQueue))
{
}

QuasarConnectorImpl::~QuasarConnectorImpl()
{
    shutdown();
}

const std::string& QuasarConnectorImpl::serviceName() const {
    return serviceName_;
}

void QuasarConnectorImpl::setMessageHandler(MessageHandler handler) {
    onQuasarMessage_ = std::move(handler);
}

void QuasarConnectorImpl::setSilentMode(bool silentMode) {
    silent_ = silentMode;
}

bool QuasarConnectorImpl::tryConnectToService()
{
    if (configuration_->getFullConfig().isMember(serviceName_)) {
        connectToService();
        return true;
    }
    YIO_LOG_DEBUG("There is no \"" << serviceName_ << "\" entry in config");
    return false;
}

void QuasarConnectorImpl::connectToService()
{
    try {
        auto port = configuration_->getServicePort(serviceName_);

        init("localhost", port, serviceName_);
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("DatacraticConnector.ConnectToService.Exception", "Fail to connect to service \"" << serviceName_ << "\": " << ex.what());
        throw;
    } catch (...) {
        YIO_LOG_ERROR_EVENT("DatacraticConnector.ConnectToService.UnknownError", "Fail to connect to service \"" << serviceName_ << "\": unexpected exception");
        throw;
    }
}

void QuasarConnectorImpl::connectToTcpHost(const std::string& hostname, int port)
{
    try {
        init(hostname, port, serviceName_);
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("DatacraticConnector.ConnectToTcpHost.Exception", "Fail to connect to \"" << serviceName_ << "\" \"" << hostname << ":" << port << "\": " << ex.what());
        throw;
    } catch (...) {
        YIO_LOG_ERROR_EVENT("DatacraticConnector.ConnectToTcpHost.UnknownError", "Fail to connect to \"" << serviceName_ << "\" \"" << hostname << ":" << port << "\": unexpected exception");
        throw;
    }
}

void QuasarConnectorImpl::sendRequest(Message& message, OnDone onDone, OnError onError, std::chrono::milliseconds timeout)
{
    message.set_request_id(makeUUID());
    if (YIO_LOG_DEBUG_ENABLED() && !silent_) {
        YIO_LOG_DEBUG("Sending quasar request: \"" << serviceName_ << "\" <- " << message.Utf8DebugString());
    }

    std::string requestData = LengthValueTokenizer::getLengthValue(message.SerializeAsString()); // Serializing request outside locks
    std::unique_lock<std::mutex> connectionLock(connectionLock_);
    std::unique_lock<std::mutex> handlersLock(handlersMutex_);
    const bool sent = TCPConnector::sendMessageUnlocked(std::move(requestData));
    if (!sent) {
        handlersLock.unlock();
        connectionLock.unlock();
        invoke(std::move(onError), "Cannot send message: not connected to " + serviceName_);
        return;
    }

    handlers_.emplace(message.request_id(), std::make_pair(std::move(onDone), std::move(onError)), timeout);
}

SharedMessage QuasarConnectorImpl::sendRequestSync(Message& message, std::chrono::milliseconds timeout)
{
    std::promise<void> requestDone;
    SharedMessage result;
    auto onDone = [&](const SharedMessage& response) {
        result = response;
        requestDone.set_value();
    };

    auto onError = [&](const std::string& errorMessage) {
        requestDone.set_exception(std::make_exception_ptr(
            std::runtime_error("Cannot send request: " + errorMessage)));
    };

    sendRequest(message, std::move(onDone), std::move(onError), timeout);

    requestDone.get_future().get();

    return result;
}

bool QuasarConnectorImpl::sendMessage(const Message& request)
{
    return TCPConnector::sendMessage(LengthValueTokenizer::getLengthValue(request.SerializeAsString()));
}

void QuasarConnectorImpl::handleMessageReceived(const std::string& message)
{
    Message protoMessage;
    if (!protoMessage.ParseFromString(TString(message))) {
        YIO_LOG_ERROR_EVENT("DatacraticConnector.HandleMessage.ProtoParseError", "Cannot parse QuasarMessage from \"" << serviceName_ << "\"");
        doError("Cannot parse QuasarMessage");
        return;
    }

    if (protoMessage.has_request_id()) {
        std::unique_lock<std::mutex> lock(handlersMutex_);
        if (handlers_.count(protoMessage.request_id())) {
            auto handlers = handlers_.pop(protoMessage.request_id());
            lock.unlock();
            invoke(std::move(handlers.first), SharedMessage{std::move(protoMessage)});
            return;
        } else {
            YIO_LOG_DEBUG("Cannot find handler for response from \"" << serviceName_ << "\" with id: '" << protoMessage.request_id() << "': {" << protoMessage.Utf8DebugString() << "}");
        }
    }

    if (onQuasarMessage_) {
        invoke(onQuasarMessage_, SharedMessage{std::move(protoMessage)});
    }
}

std::function<void()> QuasarConnectorImpl::doBeforeDisconnect(std::shared_ptr<TCPConnectionHandler> connection [[maybe_unused]])
{
    Y_VERIFY(connection);
    std::unique_lock<std::mutex> lock(handlersMutex_);
    auto uncompletedRequests = handlers_.move();
    lock.unlock();
    auto doAfterDisconnect = [=](HandlersMap& handlers) mutable {
        runErrorHandlers(handlers, "Connection has been lost");
    };

    return std::bind(doAfterDisconnect, std::move(uncompletedRequests));
}

void QuasarConnectorImpl::runErrorHandlers(HandlersMap& handlers, const std::string& errorMessage)
{
    for (auto& handler : handlers) {
        invoke(std::move(handler.second.second), errorMessage);
    }
}

template <class F, typename... Args>
void QuasarConnectorImpl::invoke(F func, Args&&... args)
{
    if (!callbackQueue_) {
        try {
            func(args...);
        } catch (const std::exception& ex) {
            YIO_LOG_ERROR_EVENT("DatacraticConnector.Invoke.Exception", "Unhandled exception in quasar connector \"" << serviceName_ << "\": " << ex.what());
            throw;
        } catch (...) {
            YIO_LOG_ERROR_EVENT("DatacraticConnector.Invoke.UnknownError", "Unexpected exception in quasar connector \"" << serviceName_ << "\"");
            throw;
        }
    } else {
        auto argsTuple = std::make_tuple(std::forward<decltype(args)>(args)...);
        auto lambda =
            [serviceName = serviceName_, func{std::move(func)}, argsTuple{std::move(argsTuple)}]() mutable {
                try {
                    std::apply(std::move(func), argsTuple);
                } catch (const std::exception& ex) {
                    YIO_LOG_ERROR_EVENT("DatacraticConnector.Invoke.Exception", "Unhandled exception in quasar connector \"" << serviceName << "\": " << ex.what());
                    throw;
                } catch (...) {
                    YIO_LOG_ERROR_EVENT("DatacraticConnector.Invoke.UnknownError", "Unexpected exception in quasar connector \"" << serviceName << "\"");
                    throw;
                }
            };
        callbackQueue_->add(std::move(lambda), tracker_);
    }
}
